#!/usr/bin/env python
# coding: utf-8

"""
Lookup for missed resources on host and mark it as BROKEN
"""

DEBUG_OUTPUT = True
FLUSH_AFTER_PRINT = True
YIELD_LIMIT = 300  # process resources simoltaneouly
CQUEUE_TASK_TIMEOUT = 4 * 60  # seconds
CQUEUE_ATTEMPTS = 5  # with progressive timeout 1 sec * attempt_no
CQUEUE_ATTEMPT_TIMEOUT = 2  # sec / progressive

RESOURCE_STATE_TO_FETCH = 'READY'

BROKEN_HOST_ERRORS = [  # skip hosts with this errors
    '[Errno 28]',  # No space left
    '[Errno 61]',  # Connection refused
    '[Errno 65]',  # Address unreachable
]

import os
import sys
import time
import socket
import datetime

import api.cqueue
from library.format import formatHosts


class Logger(object):
    def __init__(self, debug=False, flush=True):
        self.debug_level = debug
        self.flush_after_print = flush

    def _write(self, s, level):
        t = str(datetime.datetime.now())
        print '{0} {1} {2}'.format(t, level, s)
        if self.flush_after_print:
            sys.stdout.flush()

    def info(self, s):
        self._write(s, 'INF')

    def error(self, s):
        self._write(s, 'ERR')

    def debug(self, s):
        if not self.debug_level:
            return
        self._write(s, 'DBG')

logger = Logger(DEBUG_OUTPUT, FLUSH_AFTER_PRINT)


def gen_resuources(resource_manager, resources_amount, yield_limit):
    if resources_amount is not None and resources_amount < yield_limit:
        yield_limit = resources_amount
    offset = 0
    while True:
        if resources_amount is not None and offset >= resources_amount:
            raise StopIteration
        if resources_amount is not None and offset + yield_limit > resources_amount:
            yield_limit = resources_amount - offset
        resources_list = resource_manager.list_task_resources(
            state=RESOURCE_STATE_TO_FETCH,
            omit_failed=False,
            hidden=True,
            limit=yield_limit,
            offset=offset
        )
        if not resources_list:
            break
        offset += yield_limit
        yield resources_list


class SandboxBrokenResourceCheckerRunner(object):
    def __init__(self, hosts_info):
        self.hosts_info = hosts_info

    def run(self):
        host = socket.gethostname()
        if '.' not in host:
            host = socket.getfqdn()
        host = host.split('.')[0]

        ret = []
        docs = self.hosts_info[host]
        for doc in docs:
            path = doc['path']
            exists = os.path.exists(path)
            doc['exists'] = exists
            ret.append(doc)
        return ret


def processing_func(cqueue_client, resource_manager, resources_list):
    resources_info = {}  # map: id -> {doc}
    hosts_info = {}  # map: host_name -> [{doc} ..]

    logger.debug('initialize resources docs')
    for resource in resources_list:
        _id = resource.id
        hosts = resource.get_hosts()
        doc = {}
        doc['id'] = resource.id
        doc['mtime'] = resource.formatted_time()
        doc['path'] = resource.remote_path()
        doc['hosts'] = hosts
        resources_info[_id] = doc
        for host in hosts:
            hosts_info.setdefault(host, []).append(doc)
    logger.debug('resources docs OK')

    hosts = set(hosts_info.keys())
    if not hosts:
        logger.error('EMPTY HOSTS LIST')
        return

    cqueue_task = SandboxBrokenResourceCheckerRunner(hosts_info)
    cqueue_task.marshaledModules = [
        '__main__',
    ]
    logger.debug('total {0} hosts to execute: {1}'.format(len(hosts), formatHosts(hosts)))

    ok_hosts = set()
    broken_hosts = set()
    attempts = CQUEUE_ATTEMPTS
    for i in xrange(attempts):
        # progressive timeout
        time.sleep(CQUEUE_ATTEMPT_TIMEOUT * i)

        logger.debug('cqueue.run start [{0} / {1}]'.format((i + 1), attempts))
        product = cqueue_client.run(hosts - ok_hosts - broken_hosts, cqueue_task).wait(CQUEUE_TASK_TIMEOUT)
        logger.debug('cqueue.run finish')

        errors = {}

        for host, ret, err in product:
            if err:
                err = str(err)
                errors.setdefault(err, set()).add(host)
                for errorDesc in BROKEN_HOST_ERRORS:
                    if errorDesc in err:
                        broken_hosts.add(host)
                continue
            ok_hosts.add(host)
            for doc in ret:
                _id = doc['id']
                mtime = doc['mtime']
                exists = doc['exists']
                if not exists:
                    path = doc['path']
                    logger.info('Resource {0} [{1}]: does not exist on "{2}" with path "{3}"'.format(
                        _id, mtime, host, path))
                    resources_info[_id]['hosts'].remove(host)

        if not errors:
            break

        for error, hosts in errors.iteritems():
            logger.error('"{0}" on {1} hosts: {2}'.format(
                error, len(hosts), formatHosts(hosts)))

        if not hosts - ok_hosts - broken_hosts:
            break

    logger.debug('processing results')

    for _id, doc in resources_info.iteritems():
        if not doc['hosts']:
            mtime = doc['mtime']
            logger.info('Resource {0} [{1}]: empty hosts list - broken'.format(_id, mtime))
            if mark_as_broken:
                try:
                    broken_resource = resource_manager.load(_id)
                    broken_resource.mark_broken()
                    logger.info('Resource {0} [{1}]: mark as BROKEN'.format(_id, mtime))
                except Exception as ex:
                    logger.error('Resource {0} [{1}]: {2}'.format(resource.id, mtime, str(ex)))


def main(resources_amount, mark_as_broken):

    # add yasandbox root to sys.path
    sandbox_modules_dir = os.path.dirname(os.path.dirname(os.path.abspath(__file__)))
    sys.path.insert(0, sandbox_modules_dir)

    from common import config
    from yasandbox.manager import resource_manager

    settings = config.Registry()
    logger.debug('initialize resources')
    # add projects root to sys.path
    projects_modules_dir = settings.client.tasks.code_dir
    sys.path.insert(0, projects_modules_dir)
    # init
    import common.projects_handler
    common.projects_handler.load_project_types()
    logger.debug('resources OK')

    total_count = 0
    iteration = 0

    logger.debug('initialize cqudp')
    cqueue_client = api.cqueue.Client(implementation='cqudp')
    logger.debug('cqueue OK')

    for resources_list in gen_resuources(resource_manager, resources_amount, yield_limit=YIELD_LIMIT):
        logger.debug('=' * 80)

        current_step_count = len(resources_list)
        total_count += current_step_count
        logger.info('{0} of {1} step {2} [{3}]'.format(
            total_count, resources_amount, current_step_count, (iteration + 1)))

        processing_func(cqueue_client, resource_manager, resources_list)

        iteration += 1

        logger.debug('=' * 80)

    logger.debug('FINISH')


if __name__ == '__main__':
    if len(sys.argv) < 2:
        print 'USAGE: {0} resources_amount [test/mark]'.format(sys.argv[0])
        sys.exit(1)
    resources_amount = int(sys.argv[1])
    mark_as_broken = False
    if len(sys.argv) >= 3 and sys.argv[2] == 'mark':
        mark_as_broken = True
    ret = main(resources_amount, mark_as_broken)
    sys.exit(ret)
