import re
import sys
import shutil
import pathlib
import argparse
import itertools
import subprocess
import collections

import six
import ujson
import requests
from yalibrary import makelists
from yalibrary.makelists import macro_definitions as md

from sandbox.scripts.collect_devtools_resources.common import utils
from sandbox.scripts.collect_devtools_resources.common import ya_conf

RES_ID = re.compile(r'\b(\d+)\b')
RES_URL = re.compile(r'sbr:?(\d+)')

MIN_RESOURCE_ID = 63926
MAX_RESOURCE_ID = 1 << 63 - 1


def read_ya_makes(paths):
    resources = []

    for path in paths:
        try:
            yamake = makelists.from_file(path)
        except Exception:
            continue

        queue = collections.deque([yamake])
        while True:
            if len(queue) == 0:
                break

            node = queue.popleft()

            if node.node_type == md.TYPE_MACRO and node.name == "FROM_SANDBOX":
                resources.extend(itertools.chain.from_iterable(
                    RES_ID.findall(v.key())
                    for v in node.children if type(v) == md.FromSandboxValue
                ))

            if node.node_type == md.TYPE_VALUE:
                key = node.key()
                if type(key) is str:
                    resources.extend(RES_URL.findall(key))

            for el in node.children:
                queue.append(el)

    return [r for r in (int(x) for x in resources) if MIN_RESOURCE_ID <= r <= MAX_RESOURCE_ID]


def read_confs(arc_path):
    with (arc_path / "build/ya.conf.json").open() as f:
        conf = ujson.load(f)
    with (arc_path / "ya").open() as f:
        ya_script = f.read()

    resources = []

    # Read resources from ya.conf.json
    resources.extend(ya_conf.resources(conf, fetch_sources=False))

    # Read resources from ya
    ids = re.findall(r'proxy\.sandbox\.yandex-team\.ru/(\d+)', ya_script)
    ya_resources = utils.resolve_resources(id=",".join(ids), type="PLATFORM_MAPPING", fetch_sources=False)

    for res in ya_resources.values():
        resources.append(res.id)
        try:
            resp = requests.get(res.proxy, timeout=(5, 5))
            content = resp.text
        except Exception:
            continue
        platform_mapping = ujson.loads(content)
        resources.extend(x["resource_id"] for x in platform_mapping.get("data", {}).values() if "resource_id" in x)

    return resources


def find_files(path, *patterns):
    find_bin = shutil.which("find")
    if find_bin:
        args = [find_bin, path]
        for i, pattern in enumerate(patterns):
            if i:
                args.append("-o")
            args.extend(["-name", pattern])
        data = subprocess.check_output(args, encoding="utf-8")
        return sorted(f for f in data.splitlines() if f)
    path = pathlib.Path(path)
    files = sorted(str(x) for x in itertools.chain.from_iterable(path.glob("**/" + pattern) for pattern in patterns))
    return files


def main():
    parser = argparse.ArgumentParser(prog="collect-critical-resources")
    parser.add_argument("path", type=str, default=".", help="path to local copy of Arcadia")
    args = parser.parse_args()

    arc_path = pathlib.Path(args.path).expanduser()
    if not (arc_path / ".arcadia.root").exists():
        six.print_("Could not find `.arcadia.root`", file=sys.stderr)
        sys.exit(1)

    paths = find_files(str(arc_path), 'ya.make', '*.yamake')

    resources = set()
    resources.update(read_confs(arc_path))
    resources.update(read_ya_makes(paths))

    resources = sorted(resources, key=int)
    print(ujson.dumps(resources))
