#!/usr/bin/env python

""" Sandbox task remote debugger tunnel """


from __future__ import print_function

import os
import sys
import json
import socket
import select
import getpass
import logging
import httplib
import argparse
import threading as th

import requests
import websocket

from sandbox import common
import sandbox.common.types.misc as ctm


PROXY_DEFAULT_ADDR = "proxy.sandbox.yandex-team.ru"
SANDBOX_DEFAULT_URL = "https://sandbox.yandex-team.ru"


def debugger_session(pgfh, url, host, port, headers):
    logging.info("Starting remote debugger session via %s", url)
    resp = requests.post(url, json={"target": {"host": host, "port": port}}, headers=headers)
    logging.debug("Remote debugger finished with status code %r and output:\n%s", resp.status_code, resp.text)
    cz = common.console.AnsiColorizer()
    if resp.status_code != httplib.OK:
        print(cz.red(resp.text), file=pgfh)
    else:
        print(cz.black("Remote debugger bootstrapper process finished successfully."), file=pgfh)


def handle_args():
    parser = argparse.ArgumentParser(
        formatter_class=lambda *args, **kwargs: argparse.ArgumentDefaultsHelpFormatter(*args, width=120, **kwargs),
        description=sys.modules[__name__].__doc__.strip()
    )
    parser.add_argument(
        '-v', "--verbosity",
        action="count", help="Increase output verbosity"
    )
    parser.add_argument(
        '-q', "--quiet",
        action="store_true", help="Silent or quiet mode. Don't show progress meter and runtime stages"
    )
    parser.add_argument(
        '-u', "--username",
        default=None, type=str, help="User name to request OAuth token for. Uses current system user name by default"
    )
    parser.add_argument(
        '-T', "--oauth-token",
        default=None, type=str,
        help="OAuth token to authenticate instead of SSH key or interactive mode. "
        "In case of '-' specified, the token will be taken from 'OAUTH_TOKEN' environment variable"
    )
    parser.add_argument("task", metavar="TASK", help="Task ID to debug")
    parser.add_argument(
        "port",
        metavar="IDE_PORT", nargs="?", type=int, default=6996, help="Local IDE debugg server's port number"
    )
    parser.add_argument(
        "hostport", metavar="HOST:PORT", nargs="?", help="Proxy host and port to connect developer installation"
    )

    args = parser.parse_args()
    args.username = args.username or getpass.getuser()
    args.oauth_token = os.environ[ctm.OAUTH_TOKEN_ENV_NAME] if args.oauth_token == "-" else args.oauth_token
    return args


def main():
    args = handle_args()
    cz = common.console.AnsiColorizer()
    proxy = args.hostport if args.hostport else PROXY_DEFAULT_ADDR
    url = "/".join(["/", proxy, "debugger", str(args.task)])
    headers = {ctm.HTTPHeader.USER_AGENT: "debugger_proxy@" + socket.getfqdn()}

    pgfh = open(os.devnull, "wb") if args.quiet or (args.verbosity or 0) > 0 else sys.stderr
    logfile = common.log.script_log(args.verbosity, os.path.expanduser("~/.sandbox"), "debugger_tunnel.log")
    print(cz.black(sys.modules[__name__].__doc__.strip() + ", log at '{}'".format(logfile)), file=pgfh)

    if not args.hostport:
        ct = common.console.Token(SANDBOX_DEFAULT_URL, args.username, True, pgfh)
        try:
            token = args.oauth_token if args.oauth_token and ct.check(args.oauth_token) else ct()
        except KeyboardInterrupt:
            return
        headers["Authorization"] = "OAuth " + token

    with common.console.LongOperation("Opening WebSocket to ws:" + url) as op:
        logging.info("Connecting websocket to ws:%s with as %r", url, headers[ctm.HTTPHeader.USER_AGENT])
        ws = websocket.WebSocket()
        ws.connect("ws:" + url, headers=headers)
        meta = json.loads(ws.recv())
        logging.debug("Got tunnel's end meta: %r", meta)
        if "error" in meta:
            raise RuntimeError(meta["reason"])
        op.intermediate(cz.blue("Fileserver URL: ") + cz.white(meta["fileserver"]))
        op.intermediate(cz.blue("Proxy via: ") + cz.white(":".join([meta["host"], str(meta["port"])])))
        op.intermediate(cz.blue("Backend: ") + cz.white(".".join([meta["node_id"], meta["request_id"]])))

    logging.info("Connecting local IDE at %r", args.port)
    with common.console.LongOperation("Connecting local IDE at port " + str(args.port)):
        ls = socket.socket(socket.AF_INET)
        ls.connect(("localhost", args.port))
        logging.debug("Local IDE at %r connected from %r", args.port, ls.getpeername()[1])

    print(cz.black("Starting remote debugger session via ") + cz.white("http:" + url), file=pgfh)
    ses = th.Thread(target=debugger_session, args=(pgfh, "http:" + url, meta["host"], meta["port"], headers))
    ses.daemon = True
    ses.start()

    while True:
        ready, _, _ = select.select([ws.sock, ls], [], [], 1)
        if ws.sock in ready:
            data = ws.recv()
            if not data:
                logging.debug("Web socket closed")
                print(cz.yellow("Web socket closed."), file=pgfh)
                break
            logging.debug("Sending data of %d bytes length to local socket.", len(data))
            ls.send(data)
        if ls in ready:
            data = ls.recv(0xFFF)
            if not data:
                logging.debug("Local socket closed")
                print(cz.yellow("Local socket closed."), file=pgfh)
                break
            logging.debug("Sending data of %d bytes length to web socket.", len(data))
            ws.send(data)

    print(cz.green("Closing sockets."), file=pgfh)
    logging.debug("Closing web socket.")
    ws.close()
    logging.debug("Closing local socket.")
    ls.close()


if __name__ == "__main__":
    main()
