#!/usr/bin/env python
# coding: utf-8

import sys
import logging
import functools as ft

import six

if six.PY2:
    import pathlib2 as pathlib
else:
    # noinspection PyUnresolvedReferences
    import pathlib

SANDBOX_DIR = ft.reduce(lambda p, _: p.parent, range(4), pathlib.Path(__file__).resolve())  # noqa
sys.path = ["/skynet", str(SANDBOX_DIR.parent), str(SANDBOX_DIR)] + sys.path  # noqa

from sandbox import common

D_TVM_SERVICE_ID = "d-testing"
D_API_URL = "https://d.test.yandex-team.ru/api/v1"
PROVIDER = "sandbox"
COMPUTE_UNITS_ENSEMBLE_KEY = "cpuLimitUnits"
STORAGE_UNITS_ENSEMBLE_KEY = "storageUnitsBinaryLarge"

SEGMENTATIONS = {
    "ACCOUNT_TYPE": {
        "nameEn": "account type",
        "nameRu": "тип аккаунта",
        "descriptionEn": "Account type",
        "descriptionRu": "Тип аккаунта",
        "groupingOrder": 0,
        "segments": {
            "COMPUTE": {
                "nameEn": "compute",
                "nameRu": "вычисления",
                "descriptionEn": "Segment for accounts with computing quota",
                "descriptionRu": "Сегмент для аккаунтов с вычислительной квотой",
            },
            "STORAGE": {
                "nameEn": "storage",
                "nameRu": "хранилище",
                "descriptionEn": "Segment for accounts with quota on storage",
                "descriptionRu": "Сегмент для аккаунтов с квотой на хранилище",
            },
        }
    },
    "PROJECT": {
        "nameEn": "project",
        "nameRu": "проект",
        "descriptionEn": "Project",
        "descriptionRu": "Проект",
        "groupingOrder": 1,
        "segments": {
            "GENERIC": {
                "nameEn": "generic",
                "nameRu": "общий",
                "descriptionEn": "Generic",
                "descriptionRu": "Общий",
            },
            "BROWSER": {
                "nameEn": "browser",
                "nameRu": "browser",
                "descriptionEn": "Browser",
                "descriptionRu": "Browser",
            },
            "YABS": {
                "nameEn": "YABS",
                "nameRu": "YABS",
                "descriptionEn": "YABS",
                "descriptionRu": "YABS",
            },
        },
    },
    "OS": {
        "nameEn": "OS",
        "nameRu": "ОС",
        "descriptionEn": "Operating system",
        "descriptionRu": "Операционная система",
        "groupingOrder": 2,
        "segments": {
            "LINUX": {
                "nameEn": "linux",
                "nameRu": "linux",
                "descriptionEn": "Linux",
                "descriptionRu": "Linux",
            },
            "MACOS": {
                "nameEn": "macos",
                "nameRu": "macos",
                "descriptionEn": "Mac OS",
                "descriptionRu": "Mac OS",
            },
            "WINDOWS": {
                "nameEn": "windows",
                "nameRu": "windows",
                "descriptionEn": "Windows",
                "descriptionRu": "Windows",
            },
        },
    },
    "DISK": {
        "nameEn": "disk",
        "nameRu": "диск",
        "descriptionEn": "Disk",
        "descriptionRu": "Диск",
        "groupingOrder": 3,
        "segments": {
            "SSD": {
                "nameEn": "SSD",
                "nameRu": "SSD",
                "descriptionEn": "SSD",
                "descriptionRu": "SSD",
            },
            "HDD": {
                "nameEn": "HDD",
                "nameRu": "HDD",
                "descriptionEn": "HDD",
                "descriptionRu": "HDD",
            },
        },
    },
}

RESOURCE_TYPES = {
    "computing_quota": {
        "nameEn": "computing quota",
        "nameRu": "вычислительная квота",
        "descriptionEn": "Integrational quota on computing, in CPU cores",
        "descriptionRu": "Интеграционная вычислительная квота, в ядрах CPU",
        "unitsEnsembleId": COMPUTE_UNITS_ENSEMBLE_KEY,
    },
    "storage_quota": {
        "nameEn": "storage quota",
        "nameRu": "квота в хранилище",
        "descriptionEn": "Resource storage quota in dedicated S3 bucket",
        "descriptionRu": "Квота на хранение ресурсов в выделенной корзине в S3",
        "unitsEnsembleId": STORAGE_UNITS_ENSEMBLE_KEY,
    },
}

ACCOUNTS_SPACES = {
    "compute": {
        "nameEn": "computing cluster",
        "nameRu": "вычислительный кластер",
        "descriptionEn": "Computing cluster Sandbox",
        "descriptionRu": "Вычислительный кластер Sandbox",
        "readOnly": False,
        "segments": [
            {
                "segmentationId": "ACCOUNT_TYPE",
                "segmentId": "COMPUTE",
            },
        ]
    },
    "storage": {
        "nameEn": "storage",
        "nameRu": "хранилище",
        "descriptionEn": "Resource storage Sandbox",
        "descriptionRu": "Хранилище ресурсов Sandbox",
        "readOnly": False,
        "segments": [
            {
                "segmentationId": "ACCOUNT_TYPE",
                "segmentId": "STORAGE",
            },
        ]
    },
}

RESOURCES = {
    "GENERIC_LINUX_SSD": {
        "nameEn": "generic_linux_ssd",
        "nameRu": "generic_linux_ssd",
        "descriptionEn": "Generic linux hosts with SSD",
        "descriptionRu": "Linux хосты общего назначения с SSD",
        "readOnly": False,
        "orderable": True,
        "managed": True,
        "unitsEnsembleId": COMPUTE_UNITS_ENSEMBLE_KEY,
        "allowedUnitIds": ["millicores", "cores"],
        "defaultUnitId": "cores",
        "segmentations": {
            "resourceTypeId": "computing_quota",
            "segmentations": [
                {
                    "segmentationId": "ACCOUNT_TYPE",
                    "segmentId": "COMPUTE",
                },
                {
                    "segmentationId": "PROJECT",
                    "segmentId": "GENERIC",
                },
                {
                    "segmentationId": "OS",
                    "segmentId": "LINUX",
                },
                {
                    "segmentationId": "DISK",
                    "segmentId": "SSD",
                },
            ],
        },
    },
    "GENERIC_LINUX_HDD": {
        "nameEn": "generic_linux_hdd",
        "nameRu": "generic_linux_hdd",
        "descriptionEn": "Generic linux hosts with HDD",
        "descriptionRu": "Linux хосты общего назначения с HDD",
        "readOnly": False,
        "orderable": True,
        "managed": True,
        "unitsEnsembleId": COMPUTE_UNITS_ENSEMBLE_KEY,
        "allowedUnitIds": ["millicores", "cores"],
        "defaultUnitId": "cores",
        "segmentations": {
            "resourceTypeId": "computing_quota",
            "segmentations": [
                {
                    "segmentationId": "ACCOUNT_TYPE",
                    "segmentId": "COMPUTE",
                },
                {
                    "segmentationId": "PROJECT",
                    "segmentId": "GENERIC",
                },
                {
                    "segmentationId": "OS",
                    "segmentId": "LINUX",
                },
                {
                    "segmentationId": "DISK",
                    "segmentId": "HDD",
                },
            ],
        },
    },
    "GENERIC_MACOS": {
        "nameEn": "generic_macos",
        "nameRu": "generic_macos",
        "descriptionEn": "Generic MacOS hosts",
        "descriptionRu": "MacOS хосты общего назначения",
        "readOnly": False,
        "orderable": True,
        "managed": True,
        "unitsEnsembleId": COMPUTE_UNITS_ENSEMBLE_KEY,
        "allowedUnitIds": ["millicores", "cores"],
        "defaultUnitId": "cores",
        "segmentations": {
            "resourceTypeId": "computing_quota",
            "segmentations": [
                {
                    "segmentationId": "ACCOUNT_TYPE",
                    "segmentId": "COMPUTE",
                },
                {
                    "segmentationId": "PROJECT",
                    "segmentId": "GENERIC",
                },
                {
                    "segmentationId": "OS",
                    "segmentId": "MACOS",
                },
            ],
        },
    },
    "GENERIC_WINDOWS": {
        "nameEn": "generic_windows",
        "nameRu": "generic_windows",
        "descriptionEn": "Generic Windows hosts",
        "descriptionRu": "Windows хосты общего назначения",
        "readOnly": False,
        "orderable": True,
        "managed": True,
        "unitsEnsembleId": COMPUTE_UNITS_ENSEMBLE_KEY,
        "allowedUnitIds": ["millicores", "cores"],
        "defaultUnitId": "cores",
        "segmentations": {
            "resourceTypeId": "computing_quota",
            "segmentations": [
                {
                    "segmentationId": "ACCOUNT_TYPE",
                    "segmentId": "COMPUTE",
                },
                {
                    "segmentationId": "PROJECT",
                    "segmentId": "GENERIC",
                },
                {
                    "segmentationId": "OS",
                    "segmentId": "WINDOWS",
                },
            ],
        },
    },
    "BROWSER_LINUX_SSD": {
        "nameEn": "browser_linux_ssd",
        "nameRu": "browser_linux_ssd",
        "descriptionEn": "Browser's Linux hosts with SSD",
        "descriptionRu": "Linux хосты команды Browser с SSD",
        "readOnly": False,
        "orderable": True,
        "managed": True,
        "unitsEnsembleId": COMPUTE_UNITS_ENSEMBLE_KEY,
        "allowedUnitIds": ["millicores", "cores"],
        "defaultUnitId": "cores",
        "segmentations": {
            "resourceTypeId": "computing_quota",
            "segmentations": [
                {
                    "segmentationId": "ACCOUNT_TYPE",
                    "segmentId": "COMPUTE",
                },
                {
                    "segmentationId": "PROJECT",
                    "segmentId": "BROWSER",
                },
                {
                    "segmentationId": "OS",
                    "segmentId": "LINUX",
                },
                {
                    "segmentationId": "DISK",
                    "segmentId": "SSD",
                },
            ],
        },
    },
    "BROWSER_LINUX_HDD": {
        "nameEn": "browser_linux_hdd",
        "nameRu": "browser_linux_hdd",
        "descriptionEn": "Browser's Linux hosts with HDD",
        "descriptionRu": "Linux хосты команды Browser с HDD",
        "readOnly": False,
        "orderable": True,
        "managed": True,
        "unitsEnsembleId": COMPUTE_UNITS_ENSEMBLE_KEY,
        "allowedUnitIds": ["millicores", "cores"],
        "defaultUnitId": "cores",
        "segmentations": {
            "resourceTypeId": "computing_quota",
            "segmentations": [
                {
                    "segmentationId": "ACCOUNT_TYPE",
                    "segmentId": "COMPUTE",
                },
                {
                    "segmentationId": "PROJECT",
                    "segmentId": "BROWSER",
                },
                {
                    "segmentationId": "OS",
                    "segmentId": "LINUX",
                },
                {
                    "segmentationId": "DISK",
                    "segmentId": "HDD",
                },
            ],
        },
    },
    "BROWSER_MACOS": {
        "nameEn": "browser_macos",
        "nameRu": "browser_macos",
        "descriptionEn": "Browser's MacOS hosts",
        "descriptionRu": "MacOS хосты команды Browser",
        "readOnly": False,
        "orderable": True,
        "managed": True,
        "unitsEnsembleId": COMPUTE_UNITS_ENSEMBLE_KEY,
        "allowedUnitIds": ["millicores", "cores"],
        "defaultUnitId": "cores",
        "segmentations": {
            "resourceTypeId": "computing_quota",
            "segmentations": [
                {
                    "segmentationId": "ACCOUNT_TYPE",
                    "segmentId": "COMPUTE",
                },
                {
                    "segmentationId": "PROJECT",
                    "segmentId": "BROWSER",
                },
                {
                    "segmentationId": "OS",
                    "segmentId": "MACOS",
                },
            ],
        },
    },
    "BROWSER_WINDOWS": {
        "nameEn": "browser_windows",
        "nameRu": "browser_windows",
        "descriptionEn": "Browser's Windows hosts",
        "descriptionRu": "Windows хосты команды Browser",
        "readOnly": False,
        "orderable": True,
        "managed": True,
        "unitsEnsembleId": COMPUTE_UNITS_ENSEMBLE_KEY,
        "allowedUnitIds": ["millicores", "cores"],
        "defaultUnitId": "cores",
        "segmentations": {
            "resourceTypeId": "computing_quota",
            "segmentations": [
                {
                    "segmentationId": "ACCOUNT_TYPE",
                    "segmentId": "COMPUTE",
                },
                {
                    "segmentationId": "PROJECT",
                    "segmentId": "BROWSER",
                },
                {
                    "segmentationId": "OS",
                    "segmentId": "WINDOWS",
                },
            ],
        },
    },
    "YABS_LINUX_SSD": {
        "nameEn": "yabs_linux_ssd",
        "nameRu": "yabs_linux_ssd",
        "descriptionEn": "YABS's Linux hosts with SSD",
        "descriptionRu": "Linux хосты команды YABS с SSD",
        "readOnly": False,
        "orderable": True,
        "managed": True,
        "unitsEnsembleId": COMPUTE_UNITS_ENSEMBLE_KEY,
        "allowedUnitIds": ["millicores", "cores"],
        "defaultUnitId": "cores",
        "segmentations": {
            "resourceTypeId": "computing_quota",
            "segmentations": [
                {
                    "segmentationId": "ACCOUNT_TYPE",
                    "segmentId": "COMPUTE",
                },
                {
                    "segmentationId": "PROJECT",
                    "segmentId": "YABS",
                },
                {
                    "segmentationId": "OS",
                    "segmentId": "LINUX",
                },
                {
                    "segmentationId": "DISK",
                    "segmentId": "SSD",
                },
            ],
        },
    },
    "YABS_LINUX_HDD": {
        "nameEn": "yabs_linux_hdd",
        "nameRu": "yabs_linux_hdd",
        "descriptionEn": "YABS's Linux hosts with HDD",
        "descriptionRu": "Linux хосты команды YABS с HDD",
        "readOnly": False,
        "orderable": True,
        "managed": True,
        "unitsEnsembleId": COMPUTE_UNITS_ENSEMBLE_KEY,
        "allowedUnitIds": ["millicores", "cores"],
        "defaultUnitId": "cores",
        "segmentations": {
            "resourceTypeId": "computing_quota",
            "segmentations": [
                {
                    "segmentationId": "ACCOUNT_TYPE",
                    "segmentId": "COMPUTE",
                },
                {
                    "segmentationId": "PROJECT",
                    "segmentId": "YABS",
                },
                {
                    "segmentationId": "OS",
                    "segmentId": "LINUX",
                },
                {
                    "segmentationId": "DISK",
                    "segmentId": "HDD",
                },
            ],
        },
    },
    "STORAGE": {
        "nameEn": "storage",
        "nameRu": "storage",
        "descriptionEn": "Quota on Sandbox resource storage",
        "descriptionRu": "Квота на хранение ресурсов в Sandbox",
        "readOnly": False,
        "orderable": True,
        "managed": True,
        "unitsEnsembleId": STORAGE_UNITS_ENSEMBLE_KEY,
        "allowedUnitIds": ["gibibytes", "tebibytes", "pebibytes"],
        "defaultUnitId": "tebibytes",
        "segmentations": {
            "resourceTypeId": "storage_quota",
            "segmentations": [
                {
                    "segmentationId": "ACCOUNT_TYPE",
                    "segmentId": "STORAGE",
                },
            ],
        },
    },
}


def get_segmentation_by_key(d_api, key):
    for item in d_api.resourcesDirectory.segmentations[:]["items"]:
        if item["key"] == key:
            return item


def get_segment_by_key(d_api, segmegnation_id, key):
    for item in d_api.resourcesDirectory.segmentations[segmegnation_id].segments[:]["items"]:
        if item["key"] == key:
            return item


def create_segmentations(d_api):
    mapping = {}
    for segmentation_key, segmentation in SEGMENTATIONS.items():
        item = dict(key=segmentation_key)
        for name in ("nameEn", "nameRu", "descriptionEn", "descriptionRu", "groupingOrder"):
            item[name] = segmentation[name]
        try:
            result = d_api.resourcesDirectory.segmentations(item)
        except common.rest.Client.HTTPError as ex:
            if ex.status != 409:
                raise
            result = get_segmentation_by_key(d_api, segmentation_key)
        segmentation_id = result["id"]
        mapping[segmentation_key] = dict(id=segmentation_id, segments={})
        for segment_key, segment in segmentation["segments"].items():
            item = dict(key=segment_key)
            for name in ("nameEn", "nameRu", "descriptionEn", "descriptionRu"):
                item[name] = segment[name]
            try:
                result = d_api.resourcesDirectory.segmentations[segmentation_id].segments(item)
            except common.rest.Client.HTTPError as ex:
                if ex.status != 409:
                    raise
                result = get_segment_by_key(d_api, segmentation_id, segment_key)
            mapping[segmentation_key]["segments"][segment_key] = result["id"]
    return mapping


def get_units_mapping(d_api):
    return {
        unit_ensemble["key"]: dict(
            id=unit_ensemble["id"],
            units={
                unit["key"]: unit["id"]
                for unit in unit_ensemble["units"]
            }
        )
        for unit_ensemble in d_api.unitsEnsembles[:]["items"]
    }


def get_resource_type_by_key(d_api, key):
    for item in d_api.resourcesDirectory.resourceTypes[:]["items"]:
        if item["key"] == key:
            return item


def create_resource_types(d_api, units_mapping):
    mapping = {}
    for resource_type_key, resource_type in RESOURCE_TYPES.items():
        item = dict(key=resource_type_key, unitsEnsembleId=units_mapping[resource_type["unitsEnsembleId"]]["id"])
        for name in ("nameEn", "nameRu", "descriptionEn", "descriptionRu"):
            item[name] = resource_type[name]
        try:
            result = d_api.resourcesDirectory.resourceTypes(item)
        except common.rest.Client.HTTPError as ex:
            if ex.status != 409:
                raise
            result = get_resource_type_by_key(d_api, resource_type_key)
        mapping[resource_type_key] = result["id"]
    return mapping


def get_accounts_spaces_by_key(d_api, key):
    for item in d_api.resourcesDirectory.accountsSpaces[:]["items"]:
        if item["key"] == key:
            return item


def create_accounts_spaces(d_api, segmentations_mapping):
    for accounts_space_key, accounts_space in ACCOUNTS_SPACES.items():
        item = dict(
            key=accounts_space_key,
            segments=[
                dict(
                    segmentationId=segmentations_mapping[segment["segmentationId"]]["id"],
                    segmentId=segmentations_mapping[segment["segmentationId"]]["segments"][segment["segmentId"]]
                )
                for segment in accounts_space["segments"]
            ]
        )
        for name in ("nameEn", "nameRu", "descriptionEn", "descriptionRu", "readOnly"):
            item[name] = accounts_space[name]
        try:
            d_api.resourcesDirectory.accountsSpaces(item)
        except common.rest.Client.HTTPError as ex:
            if ex.status != 409:
                raise


def create_resources(d_api, segmentations_mapping, resource_types_mapping, units_mapping):
    for resource_key, resource in RESOURCES.items():
        units_ensemble = units_mapping[resource["unitsEnsembleId"]]
        item = dict(
            key=resource_key,
            unitsEnsembleId=units_ensemble["id"],
            allowedUnitIds=[units_ensemble["units"][unit_key] for unit_key in resource["allowedUnitIds"]],
            defaultUnitId=units_ensemble["units"][resource["defaultUnitId"]],
            segmentations=dict(
                resourceTypeId=resource_types_mapping[resource["segmentations"]["resourceTypeId"]],
                segmentations=[
                    dict(
                        segmentationId=segmentations_mapping[segment["segmentationId"]]["id"],
                        segmentId=segmentations_mapping[segment["segmentationId"]]["segments"][segment["segmentId"]]
                    )
                    for segment in resource["segmentations"]["segmentations"]
                ]
            )
        )
        for name in ("nameEn", "nameRu", "descriptionEn", "descriptionRu", "readOnly", "orderable", "managed"):
            item[name] = resource[name]
        try:
            d_api.resourcesDirectory.resources(item)
        except common.rest.Client.HTTPError as ex:
            if ex.status != 409:
                raise


def create_entities():
    tvm_ticket = common.tvm.TVM().get_service_ticket([D_TVM_SERVICE_ID])[D_TVM_SERVICE_ID]
    tvm_auth = common.auth.TVMSession(tvm_ticket)
    d_api = common.rest.Client(D_API_URL, auth=tvm_auth)
    provider_id = None
    for item in d_api.providers[:]["items"]:
        if item["key"] == PROVIDER:
            provider_id = item["id"]
    assert provider_id is not None
    d_api_provider = d_api.providers[provider_id]
    segmentations_mapping = create_segmentations(d_api_provider)
    units_mapping = get_units_mapping(d_api)
    resource_types_mapping = create_resource_types(d_api_provider, units_mapping)
    create_accounts_spaces(d_api_provider, segmentations_mapping)
    create_resources(d_api_provider, segmentations_mapping, resource_types_mapping, units_mapping)


def main():
    logging.basicConfig()
    logging.root.setLevel(logging.DEBUG)
    create_entities()


if __name__ == "__main__":
    sys.exit(main())
