#!/usr/bin/env python

import sys
import uuid
import datetime as dt
import logging
import textwrap
import functools as ft
import collections

import six
import requests

if six.PY2:
    import pathlib2 as pathlib
else:
    # noinspection PyUnresolvedReferences
    import pathlib

SANDBOX_DIR = ft.reduce(lambda p, _: p.parent, range(4), pathlib.Path(__file__).resolve())  # noqa
sys.path = ["/skynet", str(SANDBOX_DIR.parent), str(SANDBOX_DIR)] + sys.path  # noqa

from sandbox import common

from sandbox.yasandbox import controller
from sandbox.yasandbox.database import mapping

qc = controller.TaskQueue.qclient

D_TVM_SERVICE_ID = "d-testing"
D_API_URL = "https://d.test.yandex-team.ru/api/v1"
PROVIDER = "sandbox"

QUOTAS_BY_POOLS = {
    "YABS_LINUX_SSD": {
        "tags": "YABS & LINUX & SSD",
        "exclusive_owners": [
            "YABS_SERVER_SANDBOX_TESTS_OWN_POOL", "YABS_SERVER_SANDBOX_TESTS", "YABS-YTSTAT", "YABS_AUTOBUDDGET"
        ],
    },
    "BROWSER_WINDOWS": {
        "tags": "BROWSER & WINDOWS",
        "exclusive_owners": ["BROWSER-INFRA"],
    },
    "BROWSER_MACOS": {
        "tags": "BROWSER & OSX",
        "exclusive_owners": ["BROWSER-INFRA"],
    },
    "BROWSER_LINUX_HDD": {
        "tags": "BROWSER & LINUX & HDD",
        "exclusive_owners": [
            "BROWSER-INFRA", "BROWSER-MERGE", "BROWSER-LOCALIZATION", "BROWSER_SPEED_INFRA",
            "BROWSER-REGRESSION-SPECIALISTS", "BROWSER", "BROWSER-AUTOTESTS-INFRA",
        ],
    },
    "BROWSER_LINUX_SSD": {
        "tags": "BROWSER & LINUX & SSD",
        "exclusive_owners": [
            "BROWSER-INFRA", "BROWSER-MERGE", "BROWSER-LOCALIZATION", "BROWSER_SPEED_INFRA",
            "BROWSER-REGRESSION-SPECIALISTS", "BROWSER", "BROWSER-AUTOTESTS-INFRA",
        ],
    },
    "GENERIC_WINDOWS": {
        "tags": "GENERIC & WINDOWS",
    },
    "GENERIC_MACOS": {
        "tags": "GENERIC & OSX",
    },
    "GENERIC_LINUX_SSD": {
        "tags": "GENERIC & LINUX & (SSD | ~DISK)",
    },
    "GENERIC_LINUX_HDD": {
        "tags": "GENERIC & LINUX & HDD",
    },
}

WINDOW_MULTIPLICATOR = 10 * 12 * 3600
DEPTH = dt.datetime(2021, 6, 23)
NOT_PROVIDED_NEW_SSD_CORES = 120 * 128 * 1000
GENERIC_LINUX_HDD = "GENERIC_LINUX_HDD"
GENERIC_LINUX_SSD = "GENERIC_LINUX_SSD"
HW_RESERVE = "HW_RESERVE"
SANDBOX = "SANDBOX"


def calculate_capacity(pool):
    return sum(client.hardware.cpu.cores for client in controller.Client.list(tags=pool["tags"])) * 1000


def calculate_owners(pool_name, parents):
    query = textwrap.dedent("""
        SELECT
          DISTINCT owner
        FROM sandbox.quotaconsumptiond
        WHERE
            pool = '{pool}'
        FORMAT JSON
    """.format(pool=pool_name))
    response = requests.get(
        "http://clickhouse-sandbox.n.yandex-team.ru",
        params=dict(query=query), verify=False
    ).json()["data"]
    QUOTAS_BY_POOLS[pool_name]["owners"] = {
        _["owner"]
        for _ in response
        if _["owner"].isupper() and _["owner"] not in parents
    }


def get_common_quotas():
    owners = set()
    for pool in QUOTAS_BY_POOLS.itervalues():
        owners.update(pool["owners"])
    return dict(filter(
        lambda (owner, quota): quota.limit is not None,
        qc.multiple_owners_quota_by_pools(list(owners), use_cores=False, return_defaults=False)[None]
    ))


def remove_extra_owners(valid_owners):
    for pool in QUOTAS_BY_POOLS.itervalues():
        pool["owners"] = pool["owners"] & valid_owners


def cores2qp(cores):
    """ millicores -> milliqp """
    return cores * WINDOW_MULTIPLICATOR // 1000000


def qp2cores(qp):
    """ milliqp -> millicores """
    return qp * 1000000 // WINDOW_MULTIPLICATOR


def get_consumption(owner):
    query = textwrap.dedent("""
        SELECT
          pool,
          quantile(0.9)(real_consumption + future_consumption) AS consumption
        FROM sandbox.quotaconsumptiond
        WHERE
            date >= '{depth}'
            AND owner = '{owner}'
        GROUP BY pool
        FORMAT JSON
    """.format(owner=owner, depth=DEPTH.strftime("%Y-%m-%d")))
    response = requests.get(
        "http://clickhouse-sandbox.n.yandex-team.ru",
        params=dict(query=query), verify=False
    ).json()["data"]
    ret = {item["pool"]: item["consumption"] or 0 for item in response}

    # fix for SSD hosts without tag SSD
    ssd_pool = "GENERIC_LINUX_SSD"
    if ssd_pool in ret:
        ret[ssd_pool] = max(ret[ssd_pool], ret.get(None, 0) - sum(c for p, c in ret.iteritems() if p is not None))

    return ret


def distribute(quota, consumption):
    total_cons = consumption.get(None)
    if not total_cons:
        return {GENERIC_LINUX_HDD: quota}
    ssd_quota = round(quota * consumption.get(GENERIC_LINUX_SSD, 0) / total_cons)
    ret = {
        GENERIC_LINUX_SSD: ssd_quota,
        GENERIC_LINUX_HDD: quota - ssd_quota
    }
    return ret


def calculate_quotas(common_quotas, consumptions):
    for owner, common_quota in common_quotas.iteritems():
        distribution = distribute(common_quota.limit, consumptions[owner])
        for pool_name, quota in distribution.iteritems():
            if quota > 2472:
                QUOTAS_BY_POOLS[pool_name].setdefault("quotas", {})[owner] = quota


def print_table(common_quotas, consumptions, pool):
    print (
        "#|\n|| **№** | **Owner** | **Pool quota, cores** | **Pool quota, QP** | **Pool consumption, QP** |"
        " **Common consumption, QP** | **Common quota, QP** | **Old ratio** | **New ratio** ||"
    )
    quotas = QUOTAS_BY_POOLS[pool].get("quotas", {})
    for i, (owner, quota) in enumerate(sorted(quotas.items(), key=lambda _: _[1], reverse=True)):
        consumption = consumptions.get(owner, {})
        common_quota = common_quotas[owner].limit if owner in common_quotas else 2472
        common_cons = consumption.get(None, 0)
        cons = consumption.get(pool, 0)
        old_ratio = float(common_cons) / common_quota
        new_ratio = (cons / quota) if quota else float("inf")
        if round(new_ratio, 3) <= round(old_ratio + .0005, 3):
            new_ratio = "!!(green){:.3f}!!".format(new_ratio)
        else:
            new_ratio = "!!(red){:.3f}!!".format(new_ratio)
        if owner == SANDBOX:
            owner = "**{}**".format(owner)
        print "|| {} | {} | {:.3f} | {:.3f} | {:.3f} | {:.3f} | {:.3f} | {:.3f} | {} ||".format(
            i + 1, owner, qp2cores(quota) / 1000, quota / 1000, cons / 1000,
            common_cons / 1000, common_quota / 1000, old_ratio, new_ratio
        )
    print "|#"


def calculate():
    logging.root.setLevel(logging.ERROR)

    parents = set(mapping.Group.objects(parent__ne=None).scalar("parent"))

    for pool_name, pool in sorted(QUOTAS_BY_POOLS.iteritems()):
        print "Calculating capacity for {}".format(pool_name)
        pool["capacity"] = calculate_capacity(pool)
    print

    for pool_name, pool in sorted(QUOTAS_BY_POOLS.iteritems()):
        print "Calculating owners for {}".format(pool_name)
        calculate_owners(pool_name, parents)
    QUOTAS_BY_POOLS[GENERIC_LINUX_HDD]["owners"].add(HW_RESERVE)
    print

    QUOTAS_BY_POOLS["GENERIC_LINUX_SSD"]["capacity"] -= NOT_PROVIDED_NEW_SSD_CORES

    common_quotas = get_common_quotas()
    special_quotas = {owner: common_quotas.pop(owner, None) for owner in (SANDBOX, HW_RESERVE)}
    remove_extra_owners(common_quotas.viewkeys())
    print "Calculating consumptions"
    consumptions = {owner: get_consumption(owner) for owner in common_quotas}
    print "Calculating quotas"
    calculate_quotas(common_quotas, consumptions)
    print

    if special_quotas.get(HW_RESERVE):
        QUOTAS_BY_POOLS[GENERIC_LINUX_HDD]["quotas"][HW_RESERVE] = special_quotas[HW_RESERVE].limit
    for pool_name, pool in QUOTAS_BY_POOLS.iteritems():
        quotas = pool.setdefault("quotas", {})
        capacity = cores2qp(pool["capacity"])
        exclusive_owners = pool.get("exclusive_owners")
        if exclusive_owners:
            total_cons = sum(consumptions.get(ex_owner, {}).get(pool_name, 0) for ex_owner in exclusive_owners)
            for ex_owner in exclusive_owners:
                quotas[ex_owner] = max(capacity * consumptions.get(ex_owner, {}).get(pool_name, 0) / total_cons, 1000)
            deviation = capacity - sum(quotas.values())
            max_quota_index = max(quotas.items(), key=lambda _: _[1])[0]
            quotas[max_quota_index] -= deviation
        else:
            provided = sum(quotas.itervalues())
            quotas[SANDBOX] = capacity - provided

    generic_linux_reserve = sum(
        QUOTAS_BY_POOLS[pool]["quotas"][SANDBOX]
        for pool in (GENERIC_LINUX_HDD, GENERIC_LINUX_SSD)
    )
    generic_linux_hdd_capacity = cores2qp(QUOTAS_BY_POOLS[GENERIC_LINUX_HDD]["capacity"])
    generic_linux_ssd_capacity = cores2qp(QUOTAS_BY_POOLS[GENERIC_LINUX_SSD]["capacity"])
    generic_linux_capacity = generic_linux_hdd_capacity + generic_linux_ssd_capacity
    generic_linux_hdd_reserve = generic_linux_reserve * generic_linux_hdd_capacity / generic_linux_capacity
    deviation = generic_linux_hdd_reserve - QUOTAS_BY_POOLS[GENERIC_LINUX_HDD]["quotas"][SANDBOX]
    donors = [
        owner
        for owner, quota in QUOTAS_BY_POOLS[GENERIC_LINUX_HDD]["quotas"].iteritems()
        if owner not in (SANDBOX, HW_RESERVE) and quota > 100000
    ]
    deviation_per_owner = deviation / len(donors)
    for owner in donors:
        QUOTAS_BY_POOLS[GENERIC_LINUX_HDD]["quotas"][owner] -= deviation_per_owner
        QUOTAS_BY_POOLS[GENERIC_LINUX_SSD]["quotas"][owner] = (
            QUOTAS_BY_POOLS[GENERIC_LINUX_SSD]["quotas"].get(owner, 0) + deviation_per_owner
        )
    QUOTAS_BY_POOLS[GENERIC_LINUX_HDD]["quotas"][SANDBOX] = generic_linux_hdd_reserve
    QUOTAS_BY_POOLS[GENERIC_LINUX_SSD]["quotas"][SANDBOX] = generic_linux_reserve - generic_linux_hdd_reserve

    for owner, quota in special_quotas.iteritems():
        common_quotas[owner] = quota
        consumptions[owner] = get_consumption(owner)

    for pool_name in sorted(QUOTAS_BY_POOLS):
        print "<{{{}".format(pool_name)
        print_table(common_quotas, consumptions, pool_name)
        print "}>\n"


def group_quotas_by_services():
    for pool_name, pool in QUOTAS_BY_POOLS.iteritems():
        services = {}
        quotas = pool["quotas"]
        for owner, quota in quotas.iteritems():
            abc_service = mapping.Group.objects.with_id(owner).abc
            services.setdefault(abc_service, {})[owner] = quota
        pool["services"] = services


def get_folder_id(d_api, service_id):
    return filter(
        lambda _: _["displayName"] == "default",
        d_api.services[service_id].folders[:]["items"]
    )[0]["id"]


def import_quotas(d_api):
    provider_id = None
    for item in d_api.providers[:]["items"]:
        if item["key"] == PROVIDER:
            provider_id = item["id"]
    assert provider_id is not None
    resources = {
        item["key"]: item["id"]
        for item in d_api.providers[provider_id].resourcesDirectory.resources[:]["items"]
    }
    quotas = collections.defaultdict(dict)
    for pool, quotas_by_services in QUOTAS_BY_POOLS.iteritems():
        resource_id = resources[pool]
        for service, groups in quotas_by_services["services"].iteritems():
            service_id = common.abc.abc_service_id(service)
            quota = 0
            for group, group_quota in groups.iteritems():
                quota += group_quota
            quotas[service_id][resource_id] = quota
    request = {
        "quotas": [
            {
                "folderId": get_folder_id(d_api, sid),
                "serviceId": sid,
                "resourceQuotas": [
                    {
                        "resourceId": rid,
                        "providerId": provider_id,
                        "quota": service_quota,
                        "quotaUnitKey": "millicores",
                        "balance": service_quota,
                        "balanceUnitKey": "millicores",
                    }
                    for rid, service_quota in service_quotas.iteritems()
                ],
            }
            for sid, service_quotas in quotas.iteritems()
        ]
    }
    response = d_api["import"]._importQuotas(request)
    import_failures = response["importFailures"]
    logging.warning("Failed imports: %s", import_failures)
    logging.info("Trying to fix data for failed imports")
    fixed_quotas = []
    for failure in import_failures:
        page_token = ""
        service_id = failure["serviceId"]
        folder_id = failure["folderId"]
        resource_quotas = []
        while True:
            current_quotas = d_api.folders[folder_id].quotas["?pageToken={}".format(page_token)][:]
            if not current_quotas["items"]:
                break
            page_token = current_quotas["nextPageToken"]
            for item in current_quotas["items"]:
                resource_id = item["resourceId"]
                quota = quotas[service_id][item["resourceId"]]
                balance = quota - (item["quota"] - item["balance"])
                resource_quotas.append({
                    "resourceId": resource_id,
                    "providerId": provider_id,
                    "quota": quota,
                    "quotaUnitKey": "millicores",
                    "balance": balance,
                    "balanceUnitKey": "millicores",
                })
        if resource_quotas:
            fixed_quotas.append({
                "folderId": folder_id,
                "serviceId": service_id,
                "resourceQuotas": resource_quotas,
            })
    response = d_api["import"]._importQuotas({"quotas": fixed_quotas})
    import_failures = response["importFailures"]
    if import_failures:
        raise Exception("Failed to import quotas: {}".format(import_failures))


def set_quotas():
    for pool, quotas_by_services in QUOTAS_BY_POOLS.iteritems():
        for _, groups in quotas_by_services["services"].iteritems():
            for group, group_quota in groups.iteritems():
                qc.set_quota(group, group_quota, pool=pool, use_cores=True)


def create_accounts(d_api):
    for _, quotas_by_services in QUOTAS_BY_POOLS.iteritems():
        for service, groups in quotas_by_services["services"].iteritems():
            service_id = common.abc.abc_service_id(service)
            for name, quota in groups.iteritems():
                group = controller.Group.get(name)
                folder_id = get_folder_id(d_api, service_id)
                group.abcd = mapping.Group.ABCD(
                    account_id=str(uuid.uuid4()),
                    folder_id=folder_id,
                )
                controller.Group.edit(group)


def main():
    logging.basicConfig()
    logging.root.setLevel(logging.DEBUG)

    calculate()

    tvm_ticket = common.tvm.TVM().get_service_ticket([D_TVM_SERVICE_ID])[D_TVM_SERVICE_ID]
    tvm_auth = common.auth.TVMSession(tvm_ticket)
    d_api = common.rest.Client(D_API_URL, auth=tvm_auth)

    import_quotas(d_api)
    set_quotas()
    create_accounts(d_api)


if __name__ == "__main__":
    sys.exit(main())
