#!/usr/bin/env python

"""
The script is to import task meta-information with all its resources from the production Sandbox installation
to the local database.
"""

import os
import sys
srcroot = os.path.dirname(os.path.dirname(os.path.dirname(os.path.abspath(__file__))))
sys.path = [srcroot, os.path.dirname(srcroot)] + sys.path

import logging
import argparse
import cPickle as pickle

import common.types.task
import common.types.resource

import common.rest

import yasandbox.manager
import yasandbox.controller
import yasandbox.database.mapping as mp


def _add_task(args):
    rest = common.rest.Client(args.url)
    task = rest.task[args.task_id[0]][:]
    resources = rest.resource.read(task_id=task["id"], limit=3000)
    logging.debug('Importing task #%d of type %r owned by %r', task["id"], task["type"], task["owner"])

    from common.projects_handler import load_project_types
    load_project_types()
    import sdk2

    nt = mp.Task()
    nt.type = task["type"]
    nt.author = task["author"]
    nt.owner = task["owner"]
    nt.priority = int(common.types.task.Priority().__setstate__((
        task["priority"]["class"], task["priority"]["subclass"]
    )))
    nt.description = task["description"]
    nt.time = mp.Task.Time()
    nt.context = str(pickle.dumps(rest.task[task["id"]].context[:]))

    nt.execution = mp.Task.Execution()
    nt.execution.status = task["status"]
    nt.execution.time = mp.Task.Execution.Time()
    nt.execution.description = task["results"]["info"]
    nt.execution.auto_restart = mp.Task.Execution.AutoRestart()
    nt.execution.disk_usage = mp.Task.Execution.DiskUsage(**{
        k: v >> 20 for k, v in task["results"]["disk_usage"].iteritems()
    })

    nt.requirements = mp.Task.Requirements()
    nt.requirements.platform = task["requirements"]["platform"]
    nt.requirements.cpu_model = task["requirements"]["cpu_model"]
    nt.requirements.host = task["requirements"]["host"]
    nt.requirements.disk_space = task["requirements"]["disk_space"] >> 20
    nt.requirements.dns = task["requirements"]["dns"]

    if task.get("release", {}).get("id"):
        release = rest.release[task["id"]][:]
        nt.release = mp.Task.Release()
        nt.release.author = release["author"]
        nt.release.status = release["type"]
        nt.release.message = mp.Task.Release.Message(subject=release["subject"], body=release["message"])

    nt.save()
    mp.Audit(task_id=nt.id, content="Imported from " + task["url"], status=task["status"]).save()

    for resource in resources["items"]:
        if sdk2.Resource[resource["type"]].service:
            continue
        resource = rest.resource[resource["id"]][:]
        logging.debug("Importing resource #%d.", resource["id"])

        # create resource
        nr = mp.Resource()
        nr.type = resource["type"]
        nr.name = resource["description"]
        nr.path = resource["file_name"]
        nr.owner = resource["owner"]
        nr.task_id = nt.id
        nr.arch = resource["arch"]
        nr.time = mp.Resource.Time()
        nr.state = resource["state"]
        nr.size = resource["size"] >> 10
        nr.md5 = resource["md5"]
        nr.skynet_id = resource["skynet_id"]
        resource["attributes"]["backup_task"] = "True"
        nr.attributes = [mp.Resource.Attribute(key=k, value=v) for k, v in resource["attributes"].iteritems()]
        nr.save()
        logging.info("Resource #%d has been successfully imported as local resource #%d.", resource["id"], nr.id)

    logging.info("Task #%s has been successfully imported as local task #%s.", task["id"], nt.id)
    return nt


def handle_args():
    parser = argparse.ArgumentParser(
        formatter_class=lambda *args, **kwargs: argparse.ArgumentDefaultsHelpFormatter(*args, width=120, **kwargs),
        description="Task import tool."
    )
    parser.add_argument(
        "-U", "--url",
        help="Sandbox REST API base URL to communicate with",
        default="https://sandbox.yandex-team.ru/api/v1.0", type=str,
    )
    parser.add_argument(
        "task_id", metavar="TASK ID", nargs=1, type=int,
        help="Task ID to be imported."
    )
    return parser.parse_args()


def main(args):
    logging.basicConfig(level=logging.DEBUG, format="%(asctime)s %(levelname)4s [%(name)s] %(message)s")
    yasandbox.manager.use_locally()
    _add_task(args)
    return 0


if __name__ == "__main__":
    sys.exit(main(handle_args()))
