#!/usr/bin/env python

import sys
import logging
import argparse

import six
import requests
# noinspection PyUnresolvedReferences
import progressbar as pb

if six.PY2:
    import pathlib2 as pathlib
else:
    # noinspection PyUnresolvedReferences
    import pathlib

SANDBOX_DIR = reduce(lambda p, _: p.parent, xrange(4), pathlib.Path(__file__).resolve())  # noqa
sys.path = ["/skynet", str(SANDBOX_DIR.parent), str(SANDBOX_DIR)] + sys.path  # noqa

from sandbox import common
from sandbox.common.types import misc as ctm
from sandbox.common.types import resource as ctr
from sandbox.web.api.v1.schemas import resource as resource_schemas

logging.basicConfig()
logging.root.setLevel(logging.DEBUG)


def reshare_file(resource):
    hashes = common.mds.HashCalculator()
    resp = requests.get(resource.mds.url, stream=True)
    size = resp.headers.get(ctm.HTTPHeader.CONTENT_LENGTH)
    if size is None:
        logging.error("Content-Length is not provided by S3")
        size = 0
    size = int(size)
    pbar = pb.ProgressBar(
        widgets=[
            "Downloading {}".format(resource.http.proxy),
            pb.Bar(), " ", pb.Percentage(),
            " | ", pb.FormatLabel("%(value)d/%(max)d"),
            " | ", pb.Timer(),
            " | ", pb.ETA(),
        ],
        max_value=size,
    )
    done = 0
    for chunk in resp.iter_content(chunk_size=4096):
        hashes.update(chunk)
        done += len(chunk)
        pbar.update(done)
    pbar.finish()
    hashes.stop()
    assert hashes.md5 == resource.md5
    metadata = [common.mds.schema.MDSFileMeta.create(
        key=resource.mds.key,
        type=ctr.FileType.FILE,
        path=resource.file_name,
        executable=resource.executable,
        size=done,
        md5=hashes.md5,
        sha1_blocks=list(hashes.sha1_blocks),
    )]
    new_skynet_id = common.mds.S3().skyboned_add(metadata, resource.id, namespace=resource.mds.namespace)
    assert new_skynet_id == resource.skynet_id, (new_skynet_id, resource.skynet_id)


def reshare_directory(resource):
    metadata_link = common.mds.s3_link(str(resource.id), resource.mds.namespace)
    resp = requests.get(metadata_link)
    if resp.status_code == requests.codes.NOT_FOUND:
        resp = requests.get(resource.mds.url)
    metadata = [common.mds.schema.MDSFileMeta.create(**item) for item in resp.json()]
    new_skynet_id = common.mds.S3().skyboned_add(metadata, resource.id, namespace=resource.mds.namespace)
    assert new_skynet_id == resource.skynet_id, (new_skynet_id, resource.skynet_id)


def reshare_resource(rid):
    api = common.rest.Client()
    resource = resource_schemas.Resource.create(**api.resource[rid][:])
    if resource.multifile:
        reshare_directory(resource)
    else:
        reshare_file(resource)


def main():
    parser = argparse.ArgumentParser(
        formatter_class=argparse.ArgumentDefaultsHelpFormatter,
        description="Script for resharing resources in MDS (S3)"
    )
    # noinspection PyTypeChecker
    parser.add_argument("id", nargs="+", type=int, help="Resource id")
    args = parser.parse_args()
    for rid in args.id:
        reshare_resource(rid)


if __name__ == "__main__":
    sys.exit(main())
