#!/usr/bin/env python

import os
import argparse
from collections import defaultdict
from collections import Counter

from api.cqueue import Client
from library.sky.hostresolver import Resolver


class CollectLogs(object):
    def __init__(self, file_name):
        self.file_name = file_name

    def run(self):
        with open(self.file_name, 'rb') as log_file:
            return log_file.read()


def collect_logs(input_file, output_file):
    client = Client(implementation='cqudp')
    storages = Resolver().resolveHosts('H@SANDBOX_STORAGE')
    result_log = ""
    with client.run(storages, CollectLogs(input_file)) as session:
        for host, result, errors in session.wait():
            if errors or not result:
                print "Can not get log from {0}\n {1}".format(host, errors)
                continue
            result_log += result + '\n'
    with open(output_file, "w") as res:
        res.write(result_log)
    print "Logs collection finished successfully"


def parse_log(log_file, xmlrpc=False, aggregate=False):
    per_host_data = defaultdict(lambda: defaultdict(list))
    data = defaultdict(list)
    requests_count = Counter()
    if not xmlrpc:
        for l in log_file:
            if l.isspace():
                continue
            d = l.split()
            func = d[6]
            ip = d[7]
            ip = ip[1:-1]
            t = float(d[8][:-2])
            per_host_data[ip][func].append(t)
            data[func].append(t)
            exclude = ['http_check']
            if not args.includexmlrpc:
                exclude += 'xmlrpc'
            if func.split('/')[-1] not in exclude:
                if aggregate:
                    splited_func = func.split("?", 1)
                    url = func
                    options = None
                    if len(splited_func) > 1:
                        url, options = splited_func
                    if options:
                        func = url + '?' + options.split('=')[0]
                requests_count[func] += 1
    else:
        for l in log_file:
            if l.isspace():
                continue
            if 'Request' in l and 'from client' in l:
                d = l.split()
                func = d[11]
                ip = d[8][1:-1]
                t = float(d[14][:-2])
                per_host_data[ip][func].append(t)
                data[func].append(t)
    return data, requests_count


def print_results(response_time, frequency, limit):
    import numpy
    print 'Response time statistics per function'.center(122, '-')
    print 'function name'.ljust(50), '|', 'count'.center(11), \
        '|', 'med'.center(11), '|', 'avg'.center(11), \
        '|', 'max'.center(11), '|', 'min'.center(11), '|'
    print '-' * 122
    total_count = 0
    for func in sorted(response_time)[:limit]:
        d = response_time[func]
        total_count += len(d)
        count = str(len(d))
        med = '{0:.2f}ms'.format(numpy.median(d))
        avg = '{0:.2f}ms'.format(numpy.mean(d))
        max_ = '{0:.2f}ms'.format(max(d))
        min_ = '{0:.2f}ms'.format(min(d))
        print func.ljust(50), '|', count.center(11), \
            '|', med.center(11), '|', avg.center(11), \
            '|', max_.center(11), '|', min_.center(11), '|'
    print '-' * 122
    total_count = str(total_count)
    print 'total'.ljust(50), '|', total_count.center(11)
    print
    if frequency:
        print 'Requests frequency'.center(122, '-')
        print 'function name'.ljust(50), '|', 'requests count'.center(15)
        print '-' * 122
    for func in sorted(frequency, key=frequency.get, reverse=True)[:limit]:
        print func.ljust(50), '|', frequency[func]
    print
    print


if __name__ == '__main__':
    parser = argparse.ArgumentParser(description="Requests response time and frequency calculation script")
    parser.add_argument('file_name', help='sandbox.log path')
    parser.add_argument('-x', '--xmlrpc', dest='xmlrpc', action='store_true',
                        help='use this option to specify xmlrpc log file')
    parser.add_argument('-g', '--group', dest='group', action='store_true',
                        help='Aggregate logs from H@SANDBOX_STORAGE')
    parser.add_argument('-a', '--aggregate', dest='aggregate', action='store_true',
                        help='Aggregate URLs with same first option for frequency results')
    parser.add_argument('-l', '--limit', type=int,
                        help='Limit the results of calculation with specified number of rows. 100 by default')
    parser.add_argument('-i', '--includexmlrpc', dest='includexmlrpc', action='store_true',
                        help='Include xmlrpc requests in frequency calculation')
    args = parser.parse_args()
    tmp_file = os.path.join(os.path.dirname(os.path.abspath(__file__)), "temporary_results.txt")
    if args.limit:
        rows_limit = args.limit
    else:
        rows_limit = 100
    if args.group:
        collect_logs(args.file_name, tmp_file)
        log = open(tmp_file, "rb")
    else:
        log = open(args.file_name, 'rb')
    response_time, frequency = parse_log(log, args.xmlrpc, args.aggregate)
    print_results(response_time, frequency, rows_limit)
    log.close()
    if args.group:
        os.remove(tmp_file)
