#!/skynet/python/bin/python

"""
A script to upgrade MongoDB 3.0 to 3.2.
Expected script run behaviour:
    cat > ./server_upgrade.py ; chmod +x ./server_upgrade.py ; ./server_upgrade.py
"""

from __future__ import print_function

import sys
import shlex
import textwrap as tw
import subprocess as sp

import py


def enable_dns64():
    print("Enabling DNS64", file=sys.stderr)
    rcfg = py.path.local("/etc/resolv.conf")
    lines = []
    patch = True
    for l in rcfg.readlines():
        if patch and l.startswith("nameserver "):
            lines.append("nameserver 2a02:6b8:0:3400::5005\n")
            patch = False
        lines.append(l)
    rcfg.write("".join(lines))


def opall(op):
    print("%s MongoDB services" % (op.upper(),), file=sys.stderr)
    method = sp.check_call if op == "start" else sp.call
    for _ in ["_".join(["mongod", str(i)]) for i in xrange(37001, 37021)]:
        if py.path.local("/lib/systemd/system/{}.service".format(_)).check(file=1):
            method(["service", _, op])


def _create_mongod_config(workdir, logdir, port, no, tmpl):
    shard_workdir = workdir.join("mongod_{}".format(port))
    shard_workdir.ensure(dir=1)
    shard_workdir.chown("mongodb", "mongodb")
    py.path.local("/etc/mongod_{}.conf".format(port)).write(tmpl.format(
        port=port, no=no, workdir=str(shard_workdir), logdir=str(logdir)
    ))


def update_configs():
    workdir = py.path.local("/ssd/mongodb")
    logdir = py.path.local("/var/log/mongodb")
    for port in xrange(37001, 37006):
        no = None
        for l in py.path.local("/etc/mongod_{}.conf".format(port)).readlines():
            parts = map(str.strip, l.split("="))
            if len(parts) < 2 or parts[0] != "replSet" or not parts[1].startswith("sandbox"):
                continue
            no = parts[1][7:]
            break
        if not no:
            raise AssertionError("Cannot parse previous version configuration file!")
        _create_mongod_config(workdir, logdir, port, no, tw.dedent("""
        net:
          port: {port}
          ipv6: true
        replication:
          oplogSizeMB: 10000
          replSetName: sandbox{no}
        storage:
          engine: "mmapv1"
          dbPath: "{workdir}"
          journal:
            enabled: false
        systemLog:
          destination: file
          path: "{logdir}/mongod_{port}.log"
          logAppend: true
      """))


def upgrade_mongodb():
    print("Upgrading MongoDB", file=sys.stderr)
    sp.check_call(shlex.split("apt-key adv --keyserver hkp://keyserver.ubuntu.com:80 --recv EA312927"))
    old_list = py.path.local("/etc/apt/sources.list.d/mongodb-org-3.0.list")
    if old_list.check(file=1):
        old_list.remove()

    py.path.local("/etc/apt/sources.list.d/mongodb-org-3.2.list").write(
        "deb http://repo.mongodb.org/apt/ubuntu xenial/mongodb-org/3.2 multiverse"
    )
    sp.check_call(shlex.split("apt update"))
    sp.check_call(shlex.split("apt install --yes mongodb-org mongodb-org-server mongodb-org-shell mongodb-org-tools"))


def main():
    # sp.call("/samogon/0/cli stopall", shell=True)
    enable_dns64()
    opall("stop")
    upgrade_mongodb()
    update_configs()
    opall("start")
    print(
        "Upgrade done. WARNING: mongos and mongod_cfg are not updated and restarted - you should do it manually!",
        file=sys.stderr
    )
    return 0


if __name__ == "__main__":
    sys.exit(main())
