#!/usr/bin/env python

"""
Script to check disks status on storage hosts.
"""

from __future__ import print_function

import re
import sys
import json
import shlex
import argparse
import collections
import subprocess as sp
import distutils.version

import py

import api.cqueue
import library.sky.hostresolver

sys.path = [str(py.path.local(__file__).join("..", "..", "..", ".."))] + sys.path

# Do not try to emulate "sandbox" namespace
DISABLE_IMPORT_HOOK = True

import sandbox.common.format as fmt
import sandbox.common.console as console

OK = "OK"
SMART_MSG_RE = re.compile(r"raw value: (\d+)")


class CheckDiskStatus(object):
    osUser = "root"
    orphanTimeout = 15
    marshaledModules = [
        __name__,
        "sandbox.__init__",
        "sandbox.common.__init__",
        "sandbox.common.auth",
        "sandbox.common.console",
        "sandbox.common.context",
        "sandbox.common.encoding",
        "sandbox.common.enum",
        "sandbox.common.format",
        "sandbox.common.import_hook",
        "sandbox.common.itertools",
        "sandbox.common.patterns",
        "sandbox.common.rest",
        "sandbox.common.system",
        "sandbox.common.types",
        "sandbox.common.upload",
    ]

    def __call__(self):
        p = sp.Popen(shlex.split("sudo -u hw-watcher hw_watcher disk extended_status"), stdout=sp.PIPE, stderr=sp.PIPE)
        stdout, stderr = p.communicate()
        if p.returncode:
            raise RuntimeError("Error calling hw_watcher: " + stderr.strip())
        data = json.loads(stdout)
        failed = sorted(data.get("failed_disks", []), key=lambda _: len(_["reason"]))
        if not failed:
            yield OK
        for disk in failed:
            yield ": ".join([disk["name"], ", ".join(disk["reason"])])


def main(args):
    hosts = list(library.sky.hostresolver.Resolver().resolveHosts(args.hosts))
    pb = console.ProgressBar("Checking hosts", len(hosts))
    res = collections.defaultdict(list)
    with api.cqueue.Client() as client:
        with client.run(hosts, CheckDiskStatus()) as session:
            for host, result, err in session.wait():
                res[host].append((result, err))
                pb.update(len(res))
    pb.finish()

    table = [["Host", "Result"], None]
    cz = console.AnsiColorizer()
    for host in sorted(res, key=distutils.version.LooseVersion):
        for result, error in res[host]:
            msg = None
            if error:
                msg = cz.magenta(error)
            elif result == OK:
                msg = cz.green(OK)
            else:
                color = cz.cyan if "dmesg: error count" in result else cz.red
                for m in SMART_MSG_RE.finditer(result):
                    color = cz.red
                    try:
                        count = int(m.group(1))
                        if count < 40:
                            color = cz.yellow
                    except (IndexError, ValueError, TypeError):
                        pass
                msg = color(result)
            table.append([host, msg])
    fmt.print_table(table)


def handle_args():
    parser = argparse.ArgumentParser(
        formatter_class=lambda *args, **kwargs: argparse.ArgumentDefaultsHelpFormatter(*args, width=120, **kwargs),
        description=sys.modules[__name__].__doc__.strip()
    )
    parser.add_argument(
        "hosts",
        metavar="HOSTS", nargs="?", default="k@sandbox_stg",
        help="Hosts to be checked, Blinov calc expression. Defaults to all storage hosts"
    )
    return parser.parse_args()


if __name__ == "__main__":
    main(handle_args())
