#!/usr/bin/env python

import argparse
import datetime
import os
import sys

sys.path[0:0] = [
    "/skynet",
    os.path.dirname(os.path.dirname(os.path.dirname(os.path.abspath(__file__))))
]

from common import console
from yasandbox import controller


CHUNK_SIZE = 100


class Error(Exception):
    pass


def parse_args():
    parser = argparse.ArgumentParser(
        formatter_class=argparse.RawTextHelpFormatter,
        description=(
            "Prolong users tokens ttl in `user` and `oauth_token` collections to (current UTC time + delta) "
            "to prevent Sandbox server from doing it through Blackbox.\n"
            "Use this script ONLY in case of problems with connection to blackbox.yandex-team.ru"
        )
    )
    parser.add_argument("user_login", metavar="USER/LOGIN", nargs="*", type=str, help="List of users logins")
    parser.add_argument("-a", "--all", default=False, action="store_true", help="Update all known users tokens")
    parser.add_argument("--delta", metavar="MINUTES", type=int, default=0, help="Add it to current time")
    return parser.parse_args()


def update_ttls(user_list, all_users, target_time):
    if user_list:
        logins = user_list
    elif all_users:
        with console.LongOperation("Fetching logins from DB."):
            logins = list(controller.User.Model.objects.scalar("login"))
    else:
        raise Error("Incorrect args. See help for more.")
    with console.LongOperation("Counting amount of compatible documents"):
        count = controller.User.Model.objects(login__in=logins).count()
    if not count:
        raise Error("There is no users with specified logins or collections are empty")

    pbar = console.ProgressBar('Updating data', count)
    updated = 0
    with console.LongOperation("Updating tokens"):
        for start_ind in xrange(0, len(logins), CHUNK_SIZE):
            chunk = logins[start_ind:start_ind + CHUNK_SIZE]
            controller.OAuthCache.refresh_all(chunk, target_time)
            controller.User.refresh_staff(chunk, target_time)
            updated += len(chunk)
            pbar.update(updated)
    pbar.finish()
    return count


if __name__ == "__main__":
    controller.initialize()
    input_args = parse_args()
    cz = console.AnsiColorizer()
    try:
        target_time = datetime.datetime.utcnow() + datetime.timedelta(minutes=input_args.delta)
        cnt = update_ttls(input_args.user_login, input_args.all, target_time)
    except Error as error:
        print(cz.colorize(error, "red"))
        sys.exit(1)
    else:
        print(cz.colorize("Successfully updated {} tokens".format(cnt), "green"))
