#!/usr/bin/env python

"""
Script to switch write concern setting overall the cluster.
"""

from __future__ import print_function

import sys
import glob
import argparse
import subprocess as sp

import py
import yaml

import api.cqueue
import library.sky.hostresolver

srcroot = py.path.local(__file__).join("..", "..")
sys.path = [str(srcroot), str(srcroot.join(".."))] + sys.path

from sandbox import common


OK = "ok"
SAMOGON_ROOT = "/samogon/{key}"
SAMOGON_CLI = "/cli"  # Relative to Samogon root
CFG_GLOB = "/active/user/*/internal/*.cfg"  # Relative to Samogon root
SERVER_CFG = "/active/user/server_launcher/internal/server_tgz/sandbox/etc/settings.yaml"  # Relative to Samogon root


class SettingsSwitcher(object):
    osUser = "root"
    orphanTimeout = 15
    samogon_root = None
    dry_run = False
    wc = None

    NO_RESULT = object()

    @classmethod
    def key_finder(cls, v, k):
        if not isinstance(v, dict):
            return cls.NO_RESULT
        r = v.get(k, cls.NO_RESULT)
        if r is not cls.NO_RESULT:
            return r
        for _ in v.itervalues():
            r = cls.key_finder(_, k)
            if r is not cls.NO_RESULT:
                return r
        return cls.NO_RESULT

    def __call__(self):
        samogon_cli = self.samogon_root + SAMOGON_CLI
        if not self.dry_run:
            sp.call(["sh", samogon_cli, "stopall"])
        else:
            yield "STOPPING ALL SERVANTS"

        updates = 0
        candidates = list(glob.glob(self.samogon_root + CFG_GLOB)) + [self.samogon_root + SERVER_CFG]
        for cfg_name in candidates:
            try:
                with open(cfg_name, "rb") as fh:
                    cfg = yaml.safe_load(fh)
            except Exception:
                continue

            write_concern = self.key_finder(cfg, "write_concern")
            if write_concern is self.NO_RESULT or "w" not in write_concern:
                continue

            updates += 1
            write_concern["w"] = self.wc

            if self.dry_run:
                yield "WC FOUND IN %r: %r" % (cfg_name, write_concern)
            else:
                with open(cfg_name, "wb") as fh:
                    yaml.safe_dump(cfg, fh)

        if not self.dry_run:
            sp.call(["sh", samogon_cli, "startall"])
        else:
            yield "STARTING ALL SERVANTS"

        if not updates:
            raise RuntimeError("No config files were found")
        yield OK


def main(args):
    with common.console.LongOperation("Resolving hosts"):
        hosts = list(library.sky.hostresolver.Resolver().resolveHosts(args.hosts))
    pb = common.console.ProgressBar("Updating hosts", len(hosts))

    errors = []
    switcher = SettingsSwitcher()
    switcher.wc = args.w
    switcher.samogon_root = SAMOGON_ROOT.format(key=args.key)

    with api.cqueue.Client() as client:
        for chunk in common.utils.chunker(hosts, args.chunk_size):
            with client.run(chunk, switcher) as session:
                for host, res, err in session.wait():
                    if res == OK:
                        pb.add(1)
                    elif res:
                        print("HOST: %r, RES: %r" % (host, res))  # DEBUG
                    elif err:
                        pb.add(1)
                        errors.append([host, err])
    pb.finish()

    if not errors:
        return
    errors = [["Host", "Error"], None] + errors
    common.utils.print_table(errors)


def handle_args():
    parser = argparse.ArgumentParser(
        formatter_class=lambda *args, **kwargs: argparse.ArgumentDefaultsHelpFormatter(*args, width=120, **kwargs),
        description=sys.modules[__name__].__doc__.strip()
    )
    parser.add_argument(
        "--chunk-size", "-c",
        type=int, default=3,
        help="Amount of hosts to be operated in parallel"
    )
    parser.add_argument(
        "--key", "-k",
        type=int, default=0,
        help="Samogon key to be used on hosts"
    )
    parser.add_argument(
        "w",
        metavar="W", nargs="?", type=int,
        help="Write concern to be set in the configuration file"
    )
    parser.add_argument(
        "hosts",
        metavar="HOSTS", nargs="?", default="k@sandbox_server",
        help="Hosts to be checked, Blinov calc expression. Defaults to all storage hosts"
    )
    return parser.parse_args()


if __name__ == "__main__":
    main(handle_args())
