#!/usr/bin/env python

"""
The script drops database records for 'TEST_RTYSERVER_*' tasks older than 6 months and all related database records.
"""

import datetime as dt
import distutils.util

from common import console

from yasandbox import manager
from yasandbox.database import mapping


def chunker(data, size):
    while data:
        chunk, data = data[:size], data[size:]
        yield chunk


def main():
    manager.use_locally()
    with console.LongOperation("Fetching database records"):
        tasks = list(mapping.Task.objects(
            type__in=["TEST_RTYSERVER_UNIT", "TEST_RTYSERVER_FUNC"],
            time__updated__lt=dt.datetime.now() - dt.timedelta(days=6 * 30)
        ).scalar('id'))

    print("About to drop {} tasks. Minimum ID is {}, maximum: {}".format(len(tasks), min(tasks), max(tasks)))
    if not distutils.util.strtobool(raw_input("Is this OK (y/N)? ").strip() or "no"):
        print("Operation cancelled by user.")
        return

    pbar = console.ProgressBar('Dropping database records', len(tasks))
    for chunk in chunker(tasks, 1000):
        mapping.Task.objects(id__in=chunk).delete()
        mapping.Audit.objects(task_id__in=chunk).delete()
        mapping.Resource.objects(task_id__in=chunk).delete()
        pbar.add(len(chunk))
    pbar.finish()


if __name__ == '__main__':
    main()
