#!/usr/bin/env python

"""
Finds processes on sandbox clients that have been executing longer than currently executing task.
"""

import re
import os
import sys
import getpass
import argparse
import subprocess

# Automagically add skynet and sandbox core to the libraries search path
sys.path[0:0] = ["/skynet", os.path.dirname(os.path.dirname(os.path.abspath(__file__)))]


class PsParser(object):
    COLUMNS = "uname,pid,lstart,args"
    IGNORE_RE = ["^-liner-"]
    BREAK_RE = [r"^\[sandbox\]"]
    EXEC_RE = "^\[sandbox\].*#(?P<task_id>\d+)"
    IGNORE_EXEC_RE = ["^atop"]

    def __init__(self, os_user, sandbox_user="sandbox"):
        self.osUser = os_user
        self.sandbox_user = sandbox_user

    def __call__(self):
        try:
            ps = subprocess.check_output(["ps", "-o", self.COLUMNS, "-u", self.sandbox_user, "--sort=start_time"])
        except subprocess.CalledProcessError as e:
            if e.output.startswith("USER"):  # empty ps output
                return
            raise Exception(e.output)

        lines = ps.split("\n")
        command_pos = lines[0].find("COMMAND")

        executing = []
        for line in lines[1:]:
            command = line[command_pos:]
            match = re.search(self.EXEC_RE, command)
            if match:
                executing.append(match.group("task_id"))

        for line in lines[1:]:
            command = line[command_pos:]
            if any(re.search(regex, command) for regex in self.IGNORE_RE) or (
                any(re.search(regex, command) for regex in self.IGNORE_EXEC_RE) and
                any(task_id in command for task_id in executing)
            ):
                continue
            if any(re.search(regex, command) for regex in self.BREAK_RE):
                break
            yield line


def find_stuck(args):
    from api import cqueue
    from library.sky.hostresolver import resolver

    hosts = resolver.Resolver().resolveHosts(" ".join(args.hosts))
    with cqueue.Client(implementation="cqudp") as client:
        with client.iter(hosts, PsParser(args.user)) as session:
            for host, data, err in session.wait():
                if not data or err is not None:
                    if not isinstance(err, StopIteration):
                        print("[{}] ERROR: {}".format(host, err))
                    continue
                print "[{}] {}".format(host, data)


def handle_args():
    parser = argparse.ArgumentParser(
        formatter_class=lambda *args, **kwargs: argparse.ArgumentDefaultsHelpFormatter(*args, width=120, **kwargs),
        description="Script to find stuck processes on clients."
    )
    parser.add_argument(
        "hosts", metavar="HOST/GROUP", nargs="*",
        help="Hosts and/or host groups list."
    )
    parser.add_argument(
        "-u", "--user",
        default=getpass.getuser(), type=str, help="User to be used to connect remote servers."
    )
    return parser.parse_args()


if __name__ == "__main__":
    args = handle_args()
    find_stuck(args)
    sys.exit(0)
