import socket
import collections


# Shard configuration
#   - `id` is host's identifier (hostname actually), `localhost` can be applied on any host
#   - `alias` host identifier alias (for config shards only)
#   - `no` is Mongo's replica set configuration ID (host identifier actually)
#   - `group` is virtual shards group in format (index, total), e.g. (2, 5) means 2nd group of total 5
#   - `priority` higher is better, `0` means hidden voting, `None` means hidden non-voting
#   - `dc` is DC
class DataMapLine(collections.namedtuple("DataMapLine", ("id", "no", "group", "priority", "dc"))):
    @property
    def shards_per_group(self):  # type: () -> int
        return SHARDS // self.group[1]

    @property
    def shards_index_range(self):  # type: () -> range
        shards_index_base = (self.group[0] - 1) * self.shards_per_group + 1
        return range(shards_index_base, shards_index_base + self.shards_per_group)

    def port_by_shard_index(self, shard_index):  # type: (int) -> int
        if shard_index in self.shards_index_range:
            return 37001 + self.shards_index_range.index(shard_index)
        return 0


ConfigMapLine = collections.namedtuple("ConfigMapLine", ("id", "alias", "no", "priority", "dc"))


class MongoSetup(collections.namedtuple(
    "MongoSetup", ("data_shards_map", "config_shards_map", "zk_hosts", "memory_reserve", "sandbox_db")
)):
    def find_data_shard_map_by_host(self, host_id=None):  # type: (str or None) -> DataMapLine or None
        return self._find_shard_map_by_host(self.data_shards_map, host_id)

    def find_config_shard_map_by_host(self, host_id=None):  # type: (str or None) -> ConfigMapLine or None
        return self._find_shard_map_by_host(self.config_shards_map, host_id)

    @staticmethod
    def _find_shard_map_by_host(shard_map, host_id=None):
        if not host_id:
            host_id = socket.gethostname().split(".", 1)[0]
        return next((
            map_line for map_line in shard_map
            if map_line.id in (host_id, "localhost")
        ), None)

    @staticmethod
    def get_host_id():  # type: () -> str
        return socket.gethostname().split(".", 1)[0]


PRODUCTION_SETUP = MongoSetup(
    data_shards_map=(
        # Primary/hot-standby group in SAS
        DataMapLine("sandbox-server01", 11, (1, 5), 2, "sas"),
        DataMapLine("sandbox-server02", 11, (2, 5), 2, "sas"),
        DataMapLine("sandbox-server03", 11, (3, 5), 2, "sas"),
        DataMapLine("sandbox-server04", 11, (4, 5), 2, "sas"),
        DataMapLine("sandbox-server05", 11, (5, 5), 2, "sas"),

        # Secondary group in MAN (removed)
        # DataMapLine("sandbox-server06", 13, (1, 5), 1, "man"),
        # DataMapLine("sandbox-server07", 13, (2, 5), 1, "man"),
        # DataMapLine("sandbox-server08", 13, (3, 5), 1, "man"),
        # DataMapLine("sandbox-server09", 13, (4, 5), 1, "man"),
        # DataMapLine("sandbox-server10", 13, (5, 5), 1, "man"),

        # Primary/hot-standby group in VLA
        DataMapLine("sandbox-server11", 12, (1, 5), 2, "vla"),
        DataMapLine("sandbox-server12", 12, (2, 5), 2, "vla"),
        DataMapLine("sandbox-server13", 12, (3, 5), 2, "vla"),
        DataMapLine("sandbox-server14", 12, (4, 5), 2, "vla"),
        DataMapLine("sandbox-server15", 12, (5, 5), 2, "vla"),

        # Secondary group in IVA
        DataMapLine("sandbox-server16", 16, (1, 10), 1, "iva"),
        DataMapLine("sandbox-server17", 16, (2, 10), 1, "iva"),
        DataMapLine("sandbox-server18", 16, (3, 10), 1, "iva"),
        DataMapLine("sandbox-server19", 16, (4, 10), 1, "iva"),
        DataMapLine("sandbox-server20", 16, (5, 10), 1, "iva"),
        DataMapLine("sandbox-server21", 16, (6, 10), 1, "iva"),
        DataMapLine("sandbox-server22", 16, (7, 10), 1, "iva"),
        DataMapLine("sandbox-server23", 16, (8, 10), 1, "iva"),
        DataMapLine("sandbox-server24", 16, (9, 10), 1, "iva"),
        DataMapLine("sandbox-server25", 16, (10, 10), 1, "iva"),

        # Non-voting hidden replica in VLA
        DataMapLine("sandbox-storage47", 14, (1, 1), None, "vla"),

        # Voting hidden replicas in MAN (removed)
        # DataMapLine("sandbox-storage50", 15, (1, 1), 0, "man"),
    ),
    config_shards_map=(
        ConfigMapLine("sandbox-server05", "sandbox-mongocfg1", 4, 1, "sas"),
        # ConfigMapLine("sandbox-server10", "sandbox-mongocfg2", 5, 1, "man"),
        ConfigMapLine("sandbox-server15", "sandbox-mongocfg3", 6, 1, "vla"),
        ConfigMapLine("sandbox-server20", None, 5, 1, "iva"),
        ConfigMapLine("sandbox-storage47", None, 7, None, "vla"),
        # ConfigMapLine("sandbox-storage50", None, 8, None, "man"),
    ),
    zk_hosts=[
        "sandbox-server03", "sandbox-server11", "sandbox-server25",
    ],  # `k@sandbox_zk`
    memory_reserve=100,
    sandbox_db="sandbox",
)
PREPRODUCTION_SETUP = MongoSetup(
    data_shards_map=(
        DataMapLine("sandbox-preprod15", 3, (1, 2), 1, "iva"),
        DataMapLine("sandbox-preprod16", 3, (2, 2), 1, "iva"),

        DataMapLine("sandbox-preprod17", 4, (1, 2), 1, "myt"),
        DataMapLine("sandbox-preprod20", 5, (2, 2), 1, "sas"),

        DataMapLine("sandbox-preprod21", 5, (1, 2), 2, "sas"),
        DataMapLine("sandbox-preprod22", 4, (2, 2), 2, "myt"),
    ),
    config_shards_map=(  # `k@sandbox1_mongocfg`
        ConfigMapLine("sandbox-preprod06", None, 2, 1, "sas"),
        ConfigMapLine("sandbox-preprod15", None, 3, 1, "iva"),
        ConfigMapLine("sandbox-preprod17", None, 4, 1, "myt"),
    ),
    zk_hosts=[],
    memory_reserve=10,
    sandbox_db="sandbox_restored",
)
TESTING_SETUP = MongoSetup(
    data_shards_map=(
        DataMapLine("localhost", 1, (1, 1), 1, None),
    ),
    config_shards_map=(
        ConfigMapLine("localhost", None, 1, 1, None),
    ),
    zk_hosts=[],
    memory_reserve=10,
    sandbox_db="sandbox",
)


SETUP_MAP = {
    "preprod": PREPRODUCTION_SETUP,
    "production": PRODUCTION_SETUP,
    "testing": TESTING_SETUP,
}

DCS = ["sas", "man", "vla", "myt", "iva"]
SHARDS = 20  # Total shards amount in the cluster
