#!/usr/bin/python

"""
A script to initially setup a freshly deployed Sandbox server.
"""
# Expected script run behaviour:
#     cat > ./server_deploy.py ; chmod +x ./server_deploy.py
#    /Berkanavt/skynet/bin/gosky ; ./server_deploy.py


from __future__ import print_function

import os
import sys
import shlex
import socket
import argparse
import textwrap as tw
import subprocess as sp

import mongo_setup

try:
    import py
except ImportError:
    py = None
    psutil = None


SKYNET_PYTHON = "/skynet/python/bin/python"
DNS64_NS = "nameserver 2a02:6b8:0:3400::5005"


def memory_total():  # type: () -> int
    return os.sysconf('SC_PAGE_SIZE') * os.sysconf('SC_PHYS_PAGES')  # For Linux only


def fix_huge_pages():
    print("Disabling transparent huge pages", file=sys.stderr)
    py.path.local("/etc/systemd/system/mongod-hugepage-fix.service").write(tw.dedent("""
        [Unit]
        Before=mongos.service

        [Service]
        Type=oneshot
        ExecStart=/bin/bash -c 'echo never > /sys/kernel/mm/transparent_hugepage/enabled'
        ExecStart=/bin/bash -c 'echo never > /sys/kernel/mm/transparent_hugepage/defrag'

        [Install]
        RequiredBy=mongos.service
    """))
    sp.check_call(shlex.split("systemctl daemon-reload"))
    sp.check_call(shlex.split("systemctl enable mongod-hugepage-fix"))
    sp.check_call(shlex.split("systemctl start mongod-hugepage-fix"))


def enable_dns64():
    print("Enabling DNS64", file=sys.stderr)
    rcfg = py.path.local("/etc/resolv.conf")
    lines = []
    patch = True
    for l in rcfg.readlines():
        if patch and l.startswith("nameserver "):
            if DNS64_NS in l:
                return
            lines.append(DNS64_NS + "\n")
            patch = False
        lines.append(l)
    rcfg.write("".join(lines))


def install_mongodb(upgrade_only=False):
    print("Installing MongoDB", file=sys.stderr)
    for ver in "3.4 3.6 4.0 4.2".split():
        repo = py.path.local("/etc/apt/sources.list.d/mongodb-org-{}.list".format(ver))
        if repo.check():
            repo.remove()
    key_add_process = sp.Popen(["sudo", "apt-key", "add", "-"], stdin=sp.PIPE, stdout=sp.PIPE, stderr=sp.STDOUT)
    with open('/dev/null', 'w') as devnull:
        key_add_output, _ = key_add_process.communicate(
            sp.check_output(["curl", "https://pgp.mongodb.com/server-4.4.asc"], stderr=devnull)
        )
    if key_add_process.returncode != 0:
        raise sp.CalledProcessError(key_add_process.returncode, ["sudo", "apt-key", "add", "-"], key_add_output)
    py.path.local("/etc/apt/sources.list.d/mongodb-org-4.4.list").write(
        "deb https://mirror.yandex.ru/mirrors/repo.mongodb.org/apt/ubuntu xenial/mongodb-org/4.4 multiverse"
    )
    sp.check_call(shlex.split("apt update"))
    pkgs = [
        "mongodb-org" + pkg_suffix
        for pkg_suffix in ("", "-mongos", "-server", "-shell", "-tools", "-database-tools-extra")
    ]
    try:
        sp.check_call(shlex.split("apt install --yes " + " ".join(pkgs)))
    except sp.CalledProcessError:
        sp.check_call(shlex.split("apt install --yes " + " ".join(pkgs)))  # Retry for mongodb-tools collision resolve
    if upgrade_only:
        return
    workdir = py.path.local("/ssd/mongodb")
    workdir.ensure(dir=1)
    workdir.chown("mongodb", "mongodb")
    logdir = py.path.local("/var/log/mongodb")
    logdir.ensure(dir=1)
    logdir.chown("mongodb", "mongodb")
    return workdir, logdir


def _create_mongod_config(workdir, logdir, port, arbiter, shard_number, total_shards, setup, tmpl):
    # type: (py.path, py.path, int or str, bool or None, int, int, mongo_setup.MongoSetup, str) -> None
    shard_workdir = workdir.join("mongod_{}".format(port))
    shard_workdir.ensure(dir=1)
    shard_workdir.chown("mongodb", "mongodb")
    sharding = ""
    if arbiter is False:  # data shard, not arbiter
        sharding = tw.dedent("""
            sharding:
              clusterRole: shardsvr
        """)
    elif arbiter is None:  # configuration DB
        sharding = tw.dedent("""
            sharding:
              clusterRole: configsvr
        """)

    jbool = lambda x: str(x).lower()
    shard_cache_size = ((memory_total() - (setup.memory_reserve << 30)) / total_shards) >> 30 if total_shards else 0
    py.path.local("/etc/mongod_{}.conf".format(port)).write(tmpl.format(
        port=port, no=shard_number, workdir=str(shard_workdir),
        logdir=str(logdir), cache_size=shard_cache_size,
        bind_all_ip=jbool(True), sharding=sharding.strip()
    ).lstrip())


def _create_logrotate(logdir, port, suffix=None):  # type: (py.path, int or None, str) -> None
    basename = "mongo" + ("d_" + str(suffix or port) if port else "s")
    config = py.path.local("/etc/logrotate.d/" + basename)
    config.write(tw.dedent("""
        {logdir}/{basename}.log
        {{
            daily
            rotate 14
            compress
            dateext
            missingok
            notifempty
            sharedscripts
            postrotate
                /usr/bin/mongo --port {port} admin --eval 'db.runCommand({{logRotate : 1}})'
            endscript
        }}
    """.format(logdir=logdir, basename=basename, port=port or 22222)).lstrip())
    config.chmod(0o644)


def _create_mongod_service(port):
    py.path.local("/lib/systemd/system/mongod_{}.service".format(port)).write(tw.dedent("""
        [Unit]
        Description=High-performance, schema-free document-oriented database
        After=network.target
        Documentation=https://docs.mongodb.org/manual

        [Service]
        User=mongodb
        Group=mongodb
        Restart=always
        ExecStart=/usr/bin/numactl --interleave=all /usr/bin/mongod --quiet --config /etc/mongod_{port}.conf
        # file size
        LimitFSIZE=infinity
        # cpu time
        LimitCPU=infinity
        # virtual memory size
        LimitAS=infinity
        # open files
        LimitNOFILE=1000000
        # processes/threads
        LimitNPROC=1000000
        # total threads (user+kernel)
        TasksMax=infinity
        TasksAccounting=false
        # Lower possibility to be killed by OOM
        OOMScoreAdjust=-100
        # Restart the daemon on any problems
        Restart=always

        # Recommended limits for for mongod as specified in
        # http://docs.mongodb.org/manual/reference/ulimit/#recommended-settings

        [Install]
        WantedBy=multi-user.target
    """.format(
        port=port
    )))


def _enable_mongod_service(port):
    sp.check_call(shlex.split("systemctl daemon-reload"))
    sp.check_call(shlex.split("systemctl enable mongod_" + str(port)))
    sp.check_call(shlex.split("systemctl start mongod_" + str(port)))


def install_mongod_services(map_line, setup, workdir, logdir):
    # type: (mongo_setup.DataMapLine, mongo_setup.MongoSetup, py.path, py.path) -> None
    print("Installing MongoDB services", file=sys.stderr)
    is_arbiter = map_line.priority is None
    for shard_index in map_line.shards_index_range:
        port = map_line.port_by_shard_index(shard_index)
        _create_mongod_config(
            workdir, logdir, port, is_arbiter, shard_index, map_line.shards_per_group, setup, tw.dedent("""
                {sharding}
                net:
                  port: {port}
                  ipv6: true
                  bindIpAll: {bind_all_ip}
                replication:
                  oplogSizeMB: 10000
                  replSetName: sandbox{no}
                storage:
                  dbPath: "{workdir}"
                  engine: "wiredTiger"
                  wiredTiger:
                    engineConfig:
                      cacheSizeGB: {cache_size}
                systemLog:
                  destination: file
                  path: "{logdir}/mongod_{port}.log"
                  logAppend: true
                  logRotate: reopen
            """)
        )
        _create_logrotate(logdir, port)
        _create_mongod_service(port)
        _enable_mongod_service(port)


def install_mongod_cfg_service(setup, workdir, logdir):
    # type: (mongo_setup.MongoSetup, py.path, py.path) -> None
    print("Installing MongoDB config database service", file=sys.stderr)
    port = "cfg"
    _create_mongod_config(workdir, logdir, port, None, 0, 0, setup, tw.dedent("""
        {sharding}
        replication:
          replSetName: cs_repl_set
        net:
          port: 27019
          ipv6: true
          bindIpAll: true
        storage:
          dbPath: "{workdir}"
        systemLog:
          destination: file
          path: "{logdir}/mongod_{port}.log"
          logAppend: true
          logRotate: reopen
    """))
    _create_logrotate(logdir, 27019, port)
    _create_mongod_service(port)
    _enable_mongod_service(port)


def install_mongos_service(setup, logdir):  # type: (mongo_setup.MongoSetup, py.path) -> None
    cfgs = ",".join([
        "{}.search.yandex.net:27019".format(line.alias if line.alias else line.id)
        for line in setup.config_shards_map
    ])
    py.path.local("/lib/systemd/system/mongos.service").write(tw.dedent("""
        [Unit]
        Description=High-performance, schema-free document-oriented database
        After=network.target
        Documentation=https://docs.mongodb.org/manual

        [Service]
        User=mongodb
        Group=mongodb
        ExecStart=/usr/bin/mongos {runargs}
        # file size
        LimitFSIZE=infinity
        # cpu time
        LimitCPU=infinity
        # virtual memory size
        LimitAS=infinity
        # open files
        LimitNOFILE=1000000
        # processes/threads
        LimitNPROC=1000000
        # total threads (user+kernel)
        TasksMax=infinity
        TasksAccounting=false
        # Lower possibility to be killed by OOM
        OOMScoreAdjust=-100
        # Restart the daemon on any problems
        Restart=always

        # Recommended limits for for mongod as specified in
        # http://docs.mongodb.org/manual/reference/ulimit/#recommended-settings

        [Install]
        WantedBy=multi-user.target
    """.format(
        runargs=" ".join(map(
            lambda _: "--" + _,
            (
                "ipv6", "bind_ip=127.0.0.1,::1", "port=22222",
                "logappend", "logpath=" + str(logdir.join("mongos.log")), "logRotate reopen",
                "configdb cs_repl_set/" + cfgs
            )
        ))
    )))
    _create_logrotate(logdir, None)
    sp.check_call(shlex.split("systemctl daemon-reload"))
    sp.check_call(shlex.split("systemctl enable mongos"))
    sp.check_call(shlex.split("systemctl start mongos"))


def install_redis_and_zookeeper():
    sp.check_call(shlex.split("apt install --yes redis-server redis-tools zookeeper zookeeperd"))
    sp.check_call(shlex.split("systemctl enable zookeeper"))


def enable_slb():
    print("Enabling SLB", file=sys.stderr)
    py.path.local("/etc/rc.conf.local").write(tw.dedent("""
        ipaddr=""
        adapter_options="-lro -tso"
        ya_slb_tunnel_enable="yes"
        ya_slb6_tunnel_enable="yes"
        fix_resolvconf="no"
    """))
    sp.check_call(shlex.split("/etc/network/if-up.d/ya-slb-tun restart"))


def main(args):  # type: (argparse.Namespace) -> int
    data_shards_map = args.setup.data_shards_map
    config_shards_map = args.setup.config_shards_map
    if args.exclude:
        data_shards_map = tuple(line for line in args.setup.data_shards_map if line.dc != args.exclude)
        config_shards_map = tuple(line for line in args.setup.config_shards_map if line.dc != args.exclude)
        if (
            len(data_shards_map) == len(args.setup.data_shards_map) and
            len(config_shards_map) == len(args.setup.config_shards_map)
        ):
            print("Unable to exclude replicas from dc {} (unused)".format(args.exclude), file=sys.stderr)
    hostname = socket.gethostname()
    data_shard = next((_ for _ in data_shards_map if hostname.startswith(_.id)), None)
    config_shard = next((_ for _ in config_shards_map if hostname.startswith(_.id)), None)
    is_zk_host = any(hostname.startswith(_) for _ in args.setup.zk_hosts)
    if not data_shard and not config_shard and not args.mongos_only:
        print("Unable to determine role of this server. Aborting.", file=sys.stderr)
        return -1
    try:
        if not py:
            print("Installing skynet and re-running self with skynet python", file=sys.stderr)
            sp.call(shlex.split("/Berkanavt/skynet/bin/gosky"))
            sp.call(shlex.split("skyctl stop skynet cqudp"))
            os.execv(SKYNET_PYTHON, [SKYNET_PYTHON] + sys.argv)
        if not py.path.local("/ssd").check(dir=1) and not args.mongos_only:
            print("There's no /ssd/ directory. Aborting.", file=sys.stderr)
            return -1
        enable_dns64()
        fix_huge_pages()
        if args.upgrade_binaries:
            install_mongodb(upgrade_only=True)
            return 0
        workdir, logdir = install_mongodb()
        install_mongos_service(args.setup, logdir)
        if args.mongos_only:
            return 0
        if data_shard:
            print("This node is {!r}".format(data_shard), file=sys.stderr)
            install_mongod_services(data_shard, args.setup, workdir, logdir)
        if config_shard:
            install_mongod_cfg_service(args.setup, workdir, logdir)
        if is_zk_host:
            install_redis_and_zookeeper()
        enable_slb()
    finally:
        sp.call(shlex.split("skyctl start skynet"))
    return 0


def parse_args():  # type: () -> argparse.Namespace
    parser = argparse.ArgumentParser(
        formatter_class=argparse.RawTextHelpFormatter,
        description=sys.modules[__name__].__doc__.strip()
    )
    parser.add_argument(
        "installation", metavar="INSTALLATION", type=str, help="installation type (preprod/production/testing)",
        choices=sorted(mongo_setup.SETUP_MAP)
    )
    parser.add_argument(
        "--exclude", metavar="EXCLUDE_DC", type=str, help="exclude dc from setup", choices=mongo_setup.DCS
    )
    parser.add_argument("--mongos-only", action="store_true", help="install mongos only (for non data/config replicas)")
    parser.add_argument(
        "--upgrade-binaries", action="store_true",
        help="upgrade mongo binaries without changing setup or restarting services"
    )

    args = parser.parse_args()
    args.setup = mongo_setup.SETUP_MAP[args.installation]
    return args


if __name__ == "__main__":
    sys.exit(main(parse_args()))
