import os
import argparse

import yalibrary.makelists


# Python 3 libraries macros
PY3_LIBRARY_MACROS = ["PY3_LIBRARY", "PY3_PROGRAM", "SANDBOX_PY3_TASK", "PY3TEST"]


def parse_args():
    parser = argparse.ArgumentParser(
        formatter_class=lambda *args, **kwargs: argparse.ArgumentDefaultsHelpFormatter(*args, width=120, **kwargs),
        description="Find all PY3 source in directory"
    )
    parser.add_argument(
        "-r", "--root_dir", default=None, help=(
            "Path to directory with \"projects\". "
            "If omitted, assume tasks code is already importable (sys.path / PYTHONPATH tweaks)"
        )
    )
    return parser.parse_args()


def sources(root_dir):
    py3_sources = set()
    for dir_path, _, _ in os.walk(root_dir):
        source = os.path.join(dir_path, yalibrary.makelists.MAKELIST_NAME)
        if not os.path.exists(source):
            continue

        build_list = yalibrary.makelists.from_file(source)
        nodes = list(filter(
            None, (
                build_list.find_nodes(node_name)
                for node_name in PY3_LIBRARY_MACROS
            )
        ))
        if not nodes:
            continue

        nodes = nodes[0][0]
        real_path_prefix = dir_path
        py3_sources.add(os.path.relpath(dir_path, root_dir).replace(os.sep, "."))
        srcdir_macros = nodes.find_nodes("SRCDIR")
        if srcdir_macros:
            real_path_prefix = srcdir_macros[0].get_values()[0].name

        inside_namespace = False
        for macro in ("PY_SRCS", "TEST_SRCS"):
            desired = nodes.find_nodes(macro)
            if not desired:
                continue

            values_it = iter(desired[0].get_values())
            for value in values_it:
                if value.name in ("TOP_LEVEL", "NAMESPACE") or not value.name.endswith(".py"):
                    inside_namespace = True
                    if value.name == "NAMESPACE":
                        try:
                            next(values_it)
                        except StopIteration:
                            break
                    continue

                if inside_namespace or nodes.name in PY3_LIBRARY_MACROS:
                    abs_path = os.path.join(real_path_prefix, value.name)
                    abs_path = os.path.dirname(abs_path)
                    py3_sources.add(os.path.relpath(abs_path, root_dir).replace(os.sep, "."))

    return list(sorted(py3_sources))


def main():
    options = parse_args()
    root_dir = options.root_dir if options.root_dir is not None else os.curdir
    print(",".join(sources(root_dir)))


if __name__ == "__main__":
    main()
