#!/usr/bin/env python
"""
The script will list resources, which should be available on the host, where the script is running,
and will re-share resources, which has Skynet ID while there's no any records about them in copier.

The actual sharing processes will be performed in parallel according to constants defined below.
"""

from __future__ import print_function

import os
import sys
import glob
import signal
import logging
import argparse
import tempfile
import threading
import Queue as queue
import subprocess as sp

ROOT = "/home/zomb-sandbox/"
BASE = "/samogon/0/active/user/agentr/internal"
sys.path.extend(["/skynet"] + [ROOT + _ for _ in "client agentr".split()])

import progressbar as pb

from sandbox import common

import sandbox.common.types.resource as ctr
import sandbox.common.types.database as ctd

from sandbox.yasandbox import manager
from sandbox.yasandbox.database import mapping

MAGIC = "MAGIC!"


class RemoteResharer(object):
    def __init__(self, logger):
        # Before doing anything below, patch `common.fs.chmod_for_path` method
        # to avoid unnecessary chmods on re-sharing.
        common.fs.chmod_for_path = lambda *_: True

        self.logger = logger
        logger.info("REMOTE MODE")
        os.chdir(ROOT + "/client")

        os.environ.update({
            "SANDBOX_CONFIG": glob.glob(BASE + "/*.cfg")[0],
        })
        import sandbox.agentr.config
        import sandbox.agentr.client
        self.agentr = sandbox.agentr.client.Service(self.logger.getChild("agentr"))
        self.workers = len(sandbox.agentr.config.Registry().agentr.data.buckets) or 7
        self.queue = queue.Queue(self.workers)

    def __call__(self):
        threads = [
            threading.Thread(target=self.worker, name='#{}'.format(i))
            for i in xrange(self.workers)
        ]
        map(threading.Thread.start, threads)

        def finish():
            map(self.queue.put, [None] * self.workers)
            map(threading.Thread.join, threads)

        def sighandler(signum, _):
            msg = "Caught signal #{}. Terminating...".format(signum)
            self.logger.info(msg)
            print(msg)
            finish()
            sys.exit(signum)

        signal.signal(signal.SIGINT, sighandler)
        signal.signal(signal.SIGHUP, sighandler)
        signal.signal(signal.SIGTERM, sighandler)

        print(MAGIC)
        logging.info("Handshaked with the starter.")

        while True:
            l = sys.stdin.readline()
            logging.debug("Processing {!r} command.".format(l))
            if not l:
                break
            self.queue.put(int(l.strip()))
            logging.debug("Asking for a new job.".format(l))
            print('-')  # Send "ready" signal to the starter

        logging.debug("Finishing.")
        finish()
        logging.info("Script stopped.")

    def worker(self):
        this = threading.current_thread().name
        self.logger.info("Worker %s started.", this)
        while True:
            rid = self.queue.get()
            if not rid:
                break
            self.logger.debug("Worker %s: processing resource #%s.", this, rid)
            try:
                self.agentr.reshare(rid)
                print('+')  # Send "ok" signal to the starter
            except:
                self.logger.exception("Error sharing resource #%s", rid)
                print('~')  # Send "err" signal to the starter
        self.logger.info("Worker %s stopped.", this)


class ProgressBar(pb.ProgressBar):
    class FormatLabel(pb.FormatLabel):
        def __init__(self, *args, **kwargs):
            self.mapping["failed"] = ("failed", None)
            super(ProgressBar.FormatLabel, self).__init__(*args, **kwargs)

    def __init__(self, host, maxval):
        self.failed = 0
        super(ProgressBar, self).__init__(
            widgets=[
                "Reshare {} resources: ".format(host),
                pb.Bar(), " ", pb.Percentage(),
                " | ", self.FormatLabel("%(value)d/%(max)d F:%(failed)d"),
                " | ", pb.Timer(),
                " | ", pb.ETA(),
            ],
            max_value=maxval,
        )
        self.start()

    def add(self, val=1):
        self.value += val
        self.update(self.value)

    def failure(self, val=1):
        self.failed += val
        return self.add(val)

    def data(self):
        ret = super(ProgressBar, self).data()
        ret["failed"] = self.failed
        return ret


def main():
    settings = common.config.Registry()
    parser = argparse.ArgumentParser(
        formatter_class=lambda *args, **kwargs: argparse.ArgumentDefaultsHelpFormatter(*args, width=120, **kwargs),
        description="Sandbox resources re-sharing tool."
    )
    parser.add_argument("--dump", default=False, action="store_true", help="Dump not shared resource IDs")
    parser.add_argument("--reshare", default=False, action="store_true", help="Perform actual resources re-sharing")
    parser.add_argument("--remote", default=False, action="store_true", help="[INTERNAL] Remote mode")
    parser.add_argument(
        "host",
        metavar="HOST", nargs='?',
        help='Host to be checked. Defaults to local host'
    )
    args = parser.parse_args()
    node_id = args.host.split('.')[0] if args.host else settings.this.id
    logger = common.log.setup_log(os.path.basename(__file__) + '.log', 'DEBUG')

    if args.remote:
        os.unlink(sys.argv[0])  # Remove self first )
        return RemoteResharer(logger)()

    logger.info("STARTED")
    with common.console.LongOperation("Loading resources metainfo and counting host {} resources".format(args.host)):
        import projects.resource_types as rt
        manager.use_locally()
        with mapping.switch_db(mapping.Resource, ctd.ReadPreference.SECONDARY) as Resource:
            shareable = {
                str(rtype): rtype.share
                for rtype in rt.AbstractResource
            }
            registered = {
                skyid: rid
                for skyid, rid, rtype in Resource.objects(
                    skynet_id__exists=True,
                    state=ctr.State.READY,
                    hosts_states__host=node_id,
                ).scalar("skynet_id", "id", "type") if shareable.get(rtype)
            }

    with common.console.LongOperation("Querying skybone about {} registered resources".format(len(registered))):
        cmd = ["/skynet/tools/skybone-ctl", "check-resources", '-']
        if node_id != settings.this.id:
            cmd = ['ssh', manager.client_manager.load(args.host).fqdn] + cmd
        proc = sp.Popen(cmd, stdout=sp.PIPE, stderr=sp.PIPE, stdin=sp.PIPE)
        stdout, stderr = proc.communicate('\n'.join(registered.iterkeys()))
        if proc.returncode:
            print(
                "`{}` returned {} error code. STDERR follows:\n{}".format(cmd[0], proc.returncode, stderr),
                file=sys.stderr
            )
            return -1

        unknown = set(filter(len, stdout.split('\n')))

    print(
        "Totally detected {} resources on {}, which should be shared, "
        "{} are known by skybone, {} are unknown.".format(
            len(registered), args.host, len(registered.viewkeys() ^ unknown), len(unknown)
        )
    )

    if args.dump:
        print(repr(sorted(registered[_] for _ in unknown)))
    logger.info('Resources, which are not known for skybone: %r', sorted(unknown))
    if not args.reshare or not unknown:
        return

    pbar = ProgressBar(args.host, len(unknown))

    tfname = tempfile.mktemp()
    cmd = ['cat', '>', tfname]
    if node_id != settings.this.id:
        cmd = ['ssh', manager.client_manager.load(args.host).fqdn] + cmd

    proc = sp.Popen(cmd, stdout=sp.PIPE, stderr=sp.STDOUT, stdin=sp.PIPE)
    stdout, stderr = proc.communicate(open(sys.argv[0], "r").read())
    if proc.returncode:
        print("Error bootstrapping remote script. STDOUT/STDERR follows:\n" + stdout)
        return -1

    cmd = [sys.executable, '-u', tfname, '--remote']
    if node_id != settings.this.id:
        cmd = ['ssh', manager.client_manager.load(args.host).fqdn] + cmd
    proc = sp.Popen(cmd, stdout=sp.PIPE, stderr=sp.STDOUT, stdin=sp.PIPE)

    while True:
        data = proc.stdout.readline()
        if not data.startswith("Warning: "):
            break
    if data.strip() != MAGIC:
        print("\nError handshaking with remote side, server respond {!r}. STDOUT/STDERR follows:".format(data))
        print(proc.communicate()[0])
        return -1

    def feedback():
        data = proc.stdout.readline()
        if data and data[0] in "+~":
            (pbar.add if data[0] == "+" else pbar.failure)()
        return data

    for skyid in unknown:
        proc.stdin.write("{}\n".format(registered[skyid]))
        if not feedback():
            break

    proc.stdin.close()
    while feedback():
        pass
    pbar.finish()
    proc.wait()

    return 0


if __name__ == '__main__':
    sys.exit(main())
