#!/usr/bin/env python
# coding: utf-8

"""
Lookup for missed resources on host and mark it as BROKEN
"""

DEBUG_OUTPUT = True
FLUSH_AFTER_PRINT = True

YIELD_LIMIT = 300  # process resources simoltaneouly

RESOURCE_STATE_TO_FETCH = 'BROKEN'

STORAGES_TO_RESTORE = 1  # restore resource on N storages at time

BROKEN_HOST_ERRORS = [  # skip hosts with this errors
    '[Errno 28]',  # No space left
    '[Errno 61]',  # Connection refused
    '[Errno 65]',  # Address unreachable
]

import os
import sys
import datetime

import api.copier


class Logger(object):
    def __init__(self, debug=False, flush=True):
        self.debug_level = debug
        self.flush_after_print = flush

    def _write(self, s, level):
        t = str(datetime.datetime.now())
        print '{0} {1} {2}'.format(t, level, s)
        if self.flush_after_print:
            sys.stdout.flush()

    def info(self, s):
        self._write(s, 'INF')

    def error(self, s):
        self._write(s, 'ERR')

    def debug(self, s):
        if not self.debug_level:
            return
        self._write(s, 'DBG')

logger = Logger(DEBUG_OUTPUT, FLUSH_AFTER_PRINT)


def gen_resuources(resource_manager, resources_amount, yield_limit):
    if resources_amount is not None and resources_amount < yield_limit:
        yield_limit = resources_amount
    offset = 0
    while True:
        if resources_amount is not None and offset >= resources_amount:
            raise StopIteration
        if resources_amount is not None and offset + yield_limit > resources_amount:
            yield_limit = resources_amount - offset
        resources_list = resource_manager.list_task_resources(
            state=RESOURCE_STATE_TO_FETCH,
            omit_failed=False,
            hidden=True,
            limit=yield_limit,
            offset=offset
        )
        if not resources_list:
            break
        offset += yield_limit
        yield resources_list


def processing_func(copier, task_manager, resource_manager, resources_list, mark):
    for resource in resources_list:
        _id = resource.id
        mtime = resource.formatted_time()
        if not resource.skynet_id:
            # logger.error('Resource {0} [{1}]: empty "skynet_id"'.format(_id, mtime))
            continue
        try:
            files = copier.list(resource.skynet_id, timeout=(2 * 60)).files()
            if not files or not isinstance(files, list):
                raise EnvironmentError(
                    'Empty files list received from copier for id "{0}"'.format(resource.skynet_id))
            logger.info('Resource {0} [{1}]: files list is "{2}"'.format(_id, mtime, str(files)))
        except (api.copier.CopierError, EnvironmentError, RuntimeError) as ex:
            logger.error('Resource {0} [{1}]: "{2}"'.format(_id, mtime, str(ex)))
            continue
        if mark:
            try:
                task = task_manager.load(resource.task_id)
                resource.state = 'READY'
                resource_manager.update(resource)
                try:
                    ret = task._sync_resource(resource)
                    if not ret:
                        raise RuntimeError('empty sync result')
                    logger.info('Resource {0} [{1}]: synced with "{2}"'.format(_id, mtime, ret))
                except Exception as ex:
                    resource.state = 'BROKEN'
                    resource_manager.update(resource)
                    logger.error('Resource {0} [{1}]: "{2}"'.format(_id, mtime, str(ex)))
            except Exception as ex:
                logger.error('Resource {0} [{1}]: "{2}"'.format(_id, mtime, str(ex)))


def main(resources_amount, mark):

    # add yasandbox root to sys.path
    sandbox_modules_dir = os.path.dirname(os.path.dirname(os.path.abspath(__file__)))
    sys.path.insert(0, sandbox_modules_dir)

    import common.config
    from yasandbox.manager import task_manager, resource_manager

    settings = common.config.Registry()
    logger.debug('initialize resources')
    # add projects root to sys.path
    projects_modules_dir = settings.client.tasks.code_dir
    sys.path.insert(0, projects_modules_dir)
    # init
    import common.projects_handler
    common.projects_handler.load_project_types()
    logger.debug('resources OK')

    total_count = 0
    iteration = 0

    logger.debug('initialize copier')
    copier = api.copier.Copier()
    logger.debug('copier OK')

    for resources_list in gen_resuources(resource_manager, resources_amount, yield_limit=YIELD_LIMIT):
        logger.debug('= ' * 80)

        current_step_count = len(resources_list)
        total_count += current_step_count
        logger.info('{0} of {1} step {2} [{3}]'.format(
            total_count, resources_amount, current_step_count, (iteration + 1)))

        processing_func(copier, task_manager, resource_manager, resources_list, mark)

        iteration += 1

        logger.debug('=' * 80)

    logger.debug('FINISH')


if __name__ == '__main__':
    if len(sys.argv) < 2:
        print 'USAGE: {0} resources_amount [test/mark]'.format(sys.argv[0])
        sys.exit(1)
    resources_amount = int(sys.argv[1])
    mark = False
    if len(sys.argv) >= 3 and sys.argv[2] == 'mark':
        mark = True
    ret = main(resources_amount, mark)
    sys.exit(ret)
