#!/skynet/python/bin/python
# encoding: utf-8

import os
import sys
import pwd
import grp
import glob
import time
import socket
import shutil
import logging
import argparse
import tempfile
import requests
import collections
import subprocess as sp


API_URLS = [
    "https://sandbox.yandex-team.ru:443/api/v1.0/",
    "https://www-sandbox1.n.yandex-team.ru:443/api/v1.0/",
]
PLACE_SKYNET_PATH = [  # starts with /skynet/.info/{paths.prefix}
    "/place/skynet",
    "/place/berkanavt"
]
HOSTNAME = socket.getfqdn().split(".")[0]
SAMOGON_PATH = "/opt/sandbox/db-0"
RESOURCE_TYPE = "METADATA_BACKUP"
RESOURCE_STATE = "READY"
SKYNET_CLI = "/skynet/startup/up.py"
FSTAB_PATH = "/etc/fstab"
PASSWD_PATH = "/etc/passwd"
GROUP_PATH = "/etc/group"
RUNTIME_DIR = "/opt/sandbox/run"  # common.config.{common.dirs.runtime}
AGENTR_DB_PATH = os.path.join(RUNTIME_DIR, "agentr.db")
AGENTR_MODULE_PATH = "/home/zomb-sandbox/agentr"

USERS = {  # user -> uid, gid
    "sandbox": (3831, 3831),
    "zomb-sandbox": (29684, 21710)
}


def inform(msg, end='\n'):
    sys.stderr.write(msg + end)


def get_backup(tmp_dir, api_url):
    inform("Querying API for a backup...")
    qargs = {
        "type": RESOURCE_TYPE,
        "state": RESOURCE_STATE,
        "attrs": '{{"hostname": "{}"}}'.format(HOSTNAME),
        "order": "-id",
        "limit": 1
    }

    response = requests.get(''.join((api_url, "resource")), params=qargs)
    if not response.ok:
        inform("An error has occured: {}".format(response.text))
        sys.exit(1)

    res = response.json()["items"]
    if not res:
        inform("No backups available for this host.")
        sys.exit(1)

    res = res[0]
    inform("Last backup was made at {}. Downloading...".format(res["time"]["created"]))
    sp.check_call([SKYNET_CLI, "start", "copier"])
    sp.check_call(["sky", "get", "-N", "Backbone", "-d", tmp_dir, "-pwu", res["skynet_id"]])
    inform("Backup is downloaded. Extracting to {}...".format(tmp_dir))
    sp.check_call(["tar", "-Ipixz", "-xf", os.path.join(tmp_dir, res["file_name"]), "-C", tmp_dir])
    os.remove(os.path.join(tmp_dir, res["file_name"]))

    # flatten the extracted archive, which consists of three folders
    for root, folders, files in os.walk(tmp_dir):
        if files:
            shutil.move(os.path.join(root, files[0]), tmp_dir)


def stop_all(samogon_cli):
    if os.path.exists(SAMOGON_PATH):
        if os.path.exists(samogon_cli):
            inform("Stopping all samogon servants...")
            sp.check_call(" ".join([samogon_cli, "stopall"]), shell=True)

    inform("Stopping skynet...")
    sp.check_call([SKYNET_CLI, "stop"])
    if os.path.exists(SAMOGON_PATH):
        shutil.rmtree(SAMOGON_PATH)


def restore_etc(tmp_dir):
    inform("Restoring /etc/fstab...")
    old = open(FSTAB_PATH, "r").readlines()
    new = open(os.path.join(tmp_dir, "fstab"), "r").readlines()

    with open(FSTAB_PATH, "w") as fstab:
        def filt(s):
            return "/storage" in s

        entries_to_keep = filter(lambda s: not filt(s), old)
        bucket_entries = filter(filt, new)
        buckets_count = len(bucket_entries)

        fstab.writelines(entries_to_keep)
        fstab.writelines(bucket_entries)

    inform("Restoring /etc/group and /etc/passwd...")
    for filepath, entry in zip(
            [GROUP_PATH, PASSWD_PATH],
            ["sandbox:x:3831:zomb-sandbox,sandbox", "sandbox:x:3831:3831::/home/sandbox:"]
    ):
        contents = open(filepath, "r").readlines()
        if not filter(lambda line: line.startswith("sandbox"), contents):
            with open(filepath, "a") as fd:
                fd.write("\n" + entry)

    return buckets_count


def restore_buckets(buckets_count):
    inform("Creating and mounting buckets..")
    mounts = set()
    for mf in ("/etc/mtab", "/proc/mounts"):
        with open(mf, "r") as fd:
            for line in fd:
                mount_point = line.split(" ")[1]
                if mount_point.startswith("/storage"):
                    mounts.add(mount_point)

    for i in xrange(buckets_count):
        bucket = "/storage/{}".format(i)
        if bucket not in mounts:
            if not os.path.exists(bucket):
                os.makedirs(bucket, mode=0755)
            os.chown(bucket, USERS["zomb-sandbox"][0], USERS["sandbox"][1])
            sp.check_call(["mount", bucket])


def restore_dbs(tmp_dir, copier_db_path):
    inform("Restoring skynet copier database...")
    shutil.move(os.path.join(tmp_dir, "copier.db.backup"), copier_db_path)
    os.chmod(copier_db_path, 0644)
    os.chown(copier_db_path, pwd.getpwnam("skynet").pw_uid, grp.getgrnam("skynet").gr_gid)

    inform("Restoring agentr database...")
    if not os.path.exists(RUNTIME_DIR):
        os.makedirs(RUNTIME_DIR, mode=0775)
    os.chown(RUNTIME_DIR, *USERS["sandbox"])
    shutil.move(os.path.join(tmp_dir, "agentr.db"), AGENTR_DB_PATH)
    os.chown(AGENTR_DB_PATH, USERS["zomb-sandbox"][0], grp.getgrnam("dpt_virtual_robot").gr_gid)
    os.chmod(AGENTR_DB_PATH, 0644)


def restore_symlinks(agentr_config_glob):
    inform("Waiting for agentr to start up (a matter of few minutes)", end='')
    while True:
        inform(".", end='')
        db_created = os.path.exists(os.path.join(RUNTIME_DIR, "..", "db-0"))
        socket_created = os.path.exists(os.path.join(RUNTIME_DIR, "agentr.sock"))
        if db_created and socket_created:
            break
        time.sleep(3)

    sys.path.append(AGENTR_MODULE_PATH)
    os.environ["SANDBOX_CONFIG"] = glob.glob(agentr_config_glob)[0]

    inform("\nRestoring symlinks...")
    import agentr.client as ac
    agentr = ac.Service(logging.getLogger("agentr"))
    agentr("restore_links")


def select_mode(preprod):
    key = int(preprod)
    options = collections.namedtuple("Options", ("api_url", "samogon_cli", "agentr_config_glob", "copier_db_path"))

    return options(
        api_url=API_URLS[key],
        samogon_cli=os.path.join(SAMOGON_PATH, "key_{}/cli".format(key)),
        agentr_config_glob=os.path.join(SAMOGON_PATH, "key_{}/active/user/agentr/internal/*.cfg".format(key)),
        copier_db_path=os.path.join(PLACE_SKYNET_PATH[key], "supervisor/var/copier/rbtorrent/copier.db"),
    )


def handle_args():
    parser = argparse.ArgumentParser(
        formatter_class=lambda *args, **kwargs: argparse.ArgumentDefaultsHelpFormatter(*args, width=120, **kwargs),
        description="Restore storage host from backup."
    )
    parser.add_argument(
        "--preprod", default=False, action="store_true",
        help="assume this is a pre-production host (by default, all settings are for a productional host)"
    )
    return select_mode(parser.parse_args().preprod)


def main(options):
    tmp = tempfile.mkdtemp()

    get_backup(tmp, options.api_url)
    stop_all(options.samogon_cli)
    buckets_count = restore_etc(tmp)
    restore_buckets(buckets_count)
    restore_dbs(tmp, options.copier_db_path)

    inform("Starting skynet...")
    sp.check_call([SKYNET_CLI, "start"])

    restore_symlinks(options.agentr_config_glob)
    inform("Cleaning up...")
    shutil.rmtree(tmp)


if __name__ == '__main__':
    main(handle_args())
