# coding: utf-8


'''
Sandbox daemons manager. Use with skynet's python.

Implements start,stop,restart,status actions for all sandbox daemons.
Currently supports: client.py, server.py
XXX:
While stopping client.py the group of processess from executor.py must be killed.
in order to stop executing current task.
'''


from __future__ import print_function

import os
import sys
import errno
import fcntl
import signal
import argparse
import subprocess

import pkg_resources

pkg_resources.require('psutil')
import psutil

import common.os
import common.config

root_dir = os.path.dirname(os.path.dirname(os.path.realpath(__file__)))

OPTIONS_PROGRAMS = {
    'all': ['bin/client.py', 'bin/server.py'],
    'client': ['bin/client.py'],
    'server': ['bin/server.py'],
}


class RcActionError(Exception):
    """ Error in actions """
    pass


def error(message):
    raise RcActionError(message)


def info(_):
    return 0


def script_error(message):
    print(message, file=sys.stderr)
    return 1


def script_info(message):
    print(message, file=sys.stderr)
    return 0


def rc_action(action, option):
    """ Perform action on with a given option. """

    if action not in RC_ACTIONS:
        raise RcActionError("Unknown action: %s" % action)

    if option not in OPTIONS_PROGRAMS:
        raise RcActionError("Unknown option: %s" % option)

    programs = OPTIONS_PROGRAMS[option]

    # if one or more actions failed, than all considered as failed
    # but each command will be applied.
    rc = 0
    for p in programs:
        try:
            instances = 1
            for i in xrange(instances):
                rc = max(rc, RC_ACTIONS[action](p, i))
        except Exception as e:
            return error("Unable to {} {}: {}".format(action, p, e))
    return rc


def do_start(cmd, instance=0):
    is_locked, pid = _check_pidfile(cmd, instance)
    if is_locked:
        return error("Daemon #{} {} already running with PID {}".format(instance, cmd, pid))
    program = os.path.join(root_dir, cmd)
    if cmd not in OPTIONS_PROGRAMS["server"]:
        # Do not resolve server's main script name to real path - this will prevent server from loading newer versions.
        program = os.path.realpath(program)
    args = [sys.executable]
    args.append(program)
    if instance:
        args.append('--instance=' + str(instance))
    p = subprocess.Popen(args, close_fds=True)
    p.wait()
    return p.returncode


def do_stop(cmd, instance=0):
    is_locked, pid = _check_pidfile(cmd, instance)
    if not is_locked:
        return info("Daemon #{} {} isn't running, PID was: {}".format(instance, cmd, pid))

    if common.os.User.can_root:
        def kill(sig):
            subprocess.call(["/usr/bin/sudo", "-n", "/bin/kill", str(-sig), str(pid)])
    else:
        def kill(sig):
            os.kill(pid, sig)

    if common.config.Registry().common.installation == "TEST":
        try:
            kill(signal.SIGKILL)
        except OSError:
            raise
        return 0

    # wait 10 seconds to terminate
    p = psutil.Process(pid)
    kill(signal.SIGINT)
    try:
        p.wait(timeout=10)
    except psutil.TimeoutExpired:
        try:
            kill(signal.SIGKILL)
        except OSError:
            pass
    return 0


def do_restart(cmd, instance=0):
    is_locked, pid = _check_pidfile(cmd, instance)
    if is_locked:
        try:
            do_stop(cmd, instance)
        except Exception as e:
            info("Can't stop {}: {}".format(cmd, e))
    try:
        do_start(cmd, instance)
    except Exception as e:
            return error("Can't start {}: {}".format(cmd, e))

    return 0


def do_status(cmd, instance):
    is_locked, pid = _check_pidfile(cmd, instance)
    if is_locked:
        return info("FOUND {} with pid: {}".format(cmd, pid))
    else:
        return error("NOT FOUND {} pid was: {}".format(cmd, pid))


def do_silent_status(cmd, instance):
    is_locked, pid = _check_pidfile(cmd, instance)
    if is_locked:
        return 0
    else:
        return 1


def _check_pidfile(cmd, instance):
    """
    Check pidfile status.

    RETURN:
        1 - pidfile is locked
        0 - pidfile isn't locked or doesn't exist (pid will be None)
    """
    settings = common.config.Registry()
    pid_file_path = os.path.join(settings.client.dirs.run, '%s.pid' % os.path.basename(cmd))
    if instance:
        pid_file_path += '.' + str(instance)
    pid = None
    if not os.path.exists(pid_file_path):
        return 0, None
    else:
        with open(pid_file_path) as pidfile:
            try:
                pid = pidfile.read().strip()
                if pid:
                    pid = int(pid)
            except Exception as ex:
                raise RcActionError("Can't parse PID '{!r}' from '{}': {}".format(pid, pid_file_path, str(ex)))
    res = 0, pid
    with open(pid_file_path, 'a') as pidfile:
        try:
            fcntl.lockf(pidfile, fcntl.LOCK_EX | fcntl.LOCK_NB)
        except IOError as e:
            if e.errno in (errno.EACCES, errno.EAGAIN):
                res = 1, pid
            else:
                raise
        fcntl.lockf(pidfile, fcntl.LOCK_UN)
    return res


def check_pid(PID):
    """
        проверка наличия процесса по его PID

        :param PID: идентификатор процесса
        :return: True, если такой процесс запущен
    """
    if PID < 0:
        return False
    try:
        os.kill(PID, 0)
    except OSError:
        return False
    else:
        return True


def get_params():
    """ Parse cmd args """

    parser = argparse.ArgumentParser(
        formatter_class=lambda *args, **kwargs: argparse.ArgumentDefaultsHelpFormatter(*args, width=120, **kwargs),
        description='Sandbox services control tool.'
    )
    parser.add_argument(
        "-a", "--action",
        help="Action to run. start|stop|restart|status"
    )
    parser.add_argument(
        "-o", "--option",
        help="Option specifies which program to manage. client | server | all"
    )
    args = parser.parse_args()
    return args.action, args.option


RC_ACTIONS = {
    "start": do_start,
    "stop": do_stop,
    "restart": do_restart,
    "status": do_status,
    "silent_status": do_silent_status,
}


def as_script():
    globals()['error'], globals()['info'] = script_error, script_info
    return sys.modules[__name__]


def main():
    action, option = get_params()
    return rc_action(action, option)


if __name__ == '__main__':
    as_script()
    sys.exit(main())
