#!/usr/bin/env python

from __future__ import print_function

import sys
import argparse
import datetime as dt

import py

SOURCE_ROOT = py.path.local(__file__).join("..", "..")
sys.path = [str(SOURCE_ROOT.dirname), str(SOURCE_ROOT)] + sys.path

from sandbox.common import import_hook
import_hook.setup_sandbox_namespace()

from sandbox import common
import sandbox.common.types.task as ctt
import sandbox.common.types.client as ctc
import sandbox.common.types.database as ctd

from sandbox.yasandbox import manager
from sandbox.yasandbox.database import mapping as mp
manager.use_locally()


class DayStats(object):
    duration = dt.timedelta(seconds=0)
    amount = 0

    def add(self, other):
        self.amount += other.amount
        self.duration += other.duration
        return self

    def copy(self):
        ret = self.__class__()
        ret.amount = self.amount
        ret.duration = self.duration
        return ret

    def __add__(self, other):
        return self.copy().add(other)

    def __str__(self):
        return "%d tasks totally spend %d (%s) server-hours" % (
            self.amount, self.duration.total_seconds() / 3600, common.utils.td2str(self.duration)
        )


def _types(tag):
    hosts = {_.hostname for _ in manager.client_manager.list() if tag in _.tags}
    return hosts, set(mp.TaskTypeHostsCache.objects(hosts__in=hosts).scalar("type"))


def _p2str(p):
    return None if p is None else p.Class.val2str(p.cls)[0] + p.Subclass.val2str(p.scls)[0]


def _td2hrs(x):
    return int(x.total_seconds() / 3600)


def top(args):
    since = dt.datetime.strptime(args.since, "%Y-%m-%d")
    to = dt.datetime.strptime(args.to, "%Y-%m-%d")
    hosts, types = _types(args.tag)
    print("Servers utilization statistics for period from {} to {} ({} servers, {} task types matched the tag)".format(
        common.utils.dt2str(since), common.utils.dt2str(to), len(hosts), len(types)
    ), file=sys.stderr)

    with mp.switch_db(mp.Task, ctd.ReadPreference.SECONDARY) as Task:
        tasks = Task.objects(
            time__created__gt=since, time__created__lt=to, execution__host__in=hosts,
        ).scalar(args.groupby, "priority", "time__created", "execution__time", "execution__host")
        pb = common.console.ProgressBar("Fetching data", tasks.count())

    stats = {}
    totals = {}
    for gf, prio, ct, ext, hst in tasks:
        dst = stats.setdefault(gf, {}).setdefault(ct.date(), {})
        for k, st in [(prio, dst), (None, dst), (gf, totals)]:
            st = st.setdefault(k, DayStats())
            st.amount += 1
            if ext and ext.started and ext.finished:
                st.duration += ext.finished - ext.started
        pb.add(1)
    pb.finish()

    _P, _PC, _PS = ctt.Priority, ctt.Priority.Class, ctt.Priority.Subclass
    prios = [
        None,  # Total by all priorities
        _P(_PC.SERVICE, _PS.HIGH), _P(_PC.SERVICE, _PS.NORMAL), _P(_PC.SERVICE, _PS.LOW),
        _P(_PC.BACKGROUND, _PS.HIGH), _P(_PC.BACKGROUND, _PS.NORMAL), _P(_PC.BACKGROUND, _PS.LOW),
    ]

    label = args.groupby[0].upper() + args.groupby[1:]
    days = [(since + dt.timedelta(days=i)).date() for i in xrange((to - since).days)]
    top_gfs = sorted(totals, key=lambda k: totals[k].duration, reverse=True)[:args.rows]
    data = [
        list(common.utils.chain(
            label, *([str(day)] + [None] * 6 for day in days) if not args.cumulative else []
        )) + ["Total"],
        [None] + list(common.utils.chain(map(_p2str, prios) * len(days) * int(not args.cumulative))) + [None],
        None
    ]

    grands = {}
    top_gfs.append("Grand Total")
    tds = totals[top_gfs[-1]] = DayStats()
    for gf, dp in stats.iteritems():
        for day, pp in dp.iteritems():
            for prio, ds in pp.iteritems():
                tds.add(ds)
                grands.setdefault(day, {}).setdefault(prio, DayStats()).add(ds)

    stats[top_gfs[-1]] = grands
    fmt = _td2hrs if not args.human_readable else common.utils.td2str
    for gf in top_gfs:
        ds = [
            stats[gf].get(day, {}).get(None if prio is None else int(prio), DayStats())
            for day in days
            for prio in prios
        ]
        if args.cumulative:
            ds = []
        ds.append(totals[gf])
        data.append([gf] + [fmt(_.duration) for _ in ds])
        data.append([None] + [_.amount for _ in ds])
        if gf != top_gfs[-1]:
            data.append(None)

    if args.format in ("table", "all"):
        common.utils.print_table(data)
    if args.format in ("csv", "all"):
        for row in data:
            if not row:
                continue
            print(",".join([str(_) if _ is not None else "" for _ in row]))


def main():
    # Create the top-level parser.
    parser = argparse.ArgumentParser(
        formatter_class=lambda *args, **kwargs: argparse.ArgumentDefaultsHelpFormatter(*args, width=120, **kwargs),
        description="Servers utilization statistics calculation tool."
    )
    now = dt.datetime.utcnow()
    parser.add_argument(
        "--groupby", "-g",
        choices=("type", "owner"), default="type", help="group the table by given field"
    )
    parser.add_argument(
        "--since", "-s",
        default=common.utils.dt2str(now - dt.timedelta(days=7)).split(" ")[0],
        help="task creation UTC date in ISO 8601 format to check period from"
    )
    parser.add_argument(
        "--to", "-t",
        default=common.utils.dt2str(now + dt.timedelta(hours=23, minutes=59)).split(" ")[0],
        help="task creation UTC date in ISO 8601 format to check period to"
    )
    parser.add_argument(
        "--tag", "-T",
        default=ctc.Tag.GENERIC,
        help="Client tag to be checked"
    )
    parser.add_argument(
        "--human-readable", "-H",
        default=False, action="store_true",
        help="Format cumulative time as human readable"
    )
    parser.add_argument(
        "--format", "-f",
        choices=("table", "csv", "all"), default="table",
        help="Output resulting data as formatted table, comma-separated data or both"
    )
    parser.add_argument(
        "--cumulative", "-c",
        action="store_true",
        help="Output total counts only without per-day and per-priority statistics"
    )
    parser.add_argument("--rows", "-r", default=25, type=int, help="amount of rows to show")
    parser.set_defaults(func=top)

    mp.ensure_connection()
    # Parse the args and call whatever function was selected.
    args = parser.parse_args()
    args.func(args)


if __name__ == "__main__":
    main()
