import os
import string
import logging
import argparse
import collections
import datetime as dt

import urwid

import library.sky.hostresolver

import refueller
import resolver
import views


class SandboxTopApp(object):
    PALETTE = [
        ('body', 'black', 'light gray'),
        ('focus', 'light gray', 'dark blue', 'standout'),
        ('head', 'yellow', 'black', 'standout'),
        ('foot', 'light gray', 'black'),
        ('key', 'light cyan', 'black', 'underline'),
        ('title', 'white', 'black', 'bold'),
        ('flag', 'dark gray', 'light gray'),
        ('error', 'dark red', 'light gray'),
    ]

    def __init__(self, hosts, args):
        self.logger = logging.getLogger('App')

        self.loop = None
        self.view = None
        self.current_view = None

        self.allow_update = True
        self.keep_focus = False
        self.request_source = args.source

        self.the_r = refueller.RefuellREST(args.window, hosts, poll_period=args.poll_period)
        self.resolver = resolver.Resolver()

        self.views = collections.deque([
            views.UsersView(),
            views.MethodsView(),
            views.AddrsView(self.resolver)
        ])

    def rotate_view(self, n):
        self.views.rotate(n)
        self.loop.widget = self.view = self.views[0]

    def update_view(self, force=False):
        if not (self.allow_update or force):
            return
        self.view.update(self.the_r, keep_focus=self.keep_focus, request_source=self.request_source)

    def set_periodical(self, period, callback, *args, **kwargs):
        try:
            callback(*args, **kwargs)
        except Exception:
            logging.exception("Error in periodical handler:")
        finally:
            def wrapper(loop, user_data):
                self.set_periodical(period, callback, *args, **kwargs)

            self.loop.set_alarm_in(period, wrapper)

    def run(self):
        """Run the program."""
        self.logger.info('Running')

        self.loop = urwid.MainLoop(None, self.PALETTE, unhandled_input=self.unhandled_input, handle_mouse=False)
        self.rotate_view(0)
        self.set_periodical(1.5, self.update_view)

        try:
            self.the_r.start()
            self.resolver.start()
            self.loop.run()
        # Wait fix in Urwid mainstream
        finally:
            self.loop.screen.stop()
            self.the_r.stop()
            self.resolver.stop()

        self.logger.info('Stopped')

    def unhandled_input(self, k):
        if k in ('q', 'Q'):
            raise urwid.ExitMainLoop()

        if k == 'left':
            self.rotate_view(1)
            self.update_view()

        if k == 'right':
            self.rotate_view(-1)
            self.update_view()

        if k in ('r', 'R'):
            self.update_view(force=True)

        if k == '=':
            self.the_r.window_period += dt.timedelta(seconds=1)
            self.update_view()

        if k == '-':
            self.the_r.window_period -= dt.timedelta(seconds=1)
            self.update_view()

        if k in ('a', 'A'):
            self.allow_update = not self.allow_update

        if k in ('f', 'F'):
            self.keep_focus = not self.keep_focus

        if k in string.digits:
            if self.view and self.view.sort_col(int(k)):
                self.update_view()

        if k in ('i', 'I'):
            self.the_r.requests = refueller.REQUESTS.INPROGRESS

        if k in ('l', 'l'):
            self.the_r.requests = refueller.REQUESTS.LAST


def main():
    logging.basicConfig(level=logging.DEBUG,
                        format='%(asctime)s %(name)-12s %(levelname)-8s %(message)s',
                        datefmt='%m-%d %H:%M:%S',
                        filename='stop.log',
                        filemode='w')

    requests_log = logging.getLogger("requests.packages.urllib3")
    requests_log.setLevel(logging.DEBUG)
    requests_log.propagate = False

    logging.info("Start STOP. pid: %s", os.getpid())

    parser = argparse.ArgumentParser(
        prog='stop',
        description='Sandbox TOP utility.',
        formatter_class=argparse.ArgumentDefaultsHelpFormatter
    )
    parser.add_argument('hosts', default='K@search_instrum-sandbox_server', nargs='*',
                        help='conductor group')
    parser.add_argument('--source', choices=['web', 'rpc', 'api'],
                        help='filter source')
    parser.add_argument('--window', type=int, default=15,
                        help='window size')
    parser.add_argument('--poll-period', type=int, default=2,
                        help='servers poll period')

    args = parser.parse_args()

    host_resolver = library.sky.hostresolver.Resolver()
    hosts = host_resolver.resolveHosts(args.hosts)

    SandboxTopApp(hosts, args).run()


if __name__ == '__main__':
    main()
