import time
import Queue
import socket
import threading
import datetime as dt

import collections


class Resolver(threading.Thread):
    Addr = collections.namedtuple('Address', ['hostname', 'aliaslist', 'ipaddrlist', 'updated_at', 'used_at'])

    def __init__(self):
        super(Resolver, self).__init__()
        self._queue = Queue.PriorityQueue(maxsize=1000)
        self._resolved = {}
        self._event = threading.Event()

    def get(self, ip):
        addr = self._resolved.get(ip)
        if addr:
            self._resolved[ip] = addr._replace(used_at=dt.datetime.utcnow())
            return addr
        else:
            try:
                self._queue.put_nowait(ip)
            except Queue.Full:
                pass

    def stop(self):
        self._event.set()
        self.join()

    def _check_cache(self):
        for ip, addr in sorted(self._resolved.iteritems(), key=lambda item: item[1].updated_at, reverse=True):
            if dt.datetime.utcnow() - addr.used_at > dt.timedelta(minutes=5):
                del self._resolved[ip]
            else:
                try:
                    self._queue.put_nowait(ip)
                except Queue.Full:
                    break

    def run(self):
        while not self._event.is_set():
            try:
                ip = self._queue.get_nowait()
            except Queue.Empty:
                if self._resolved:
                    self._check_cache()
                else:
                    time.sleep(0.5)
                continue

            try:
                r = socket.gethostbyaddr(ip)
            except Exception:
                continue

            addr = self._resolved.get(ip)
            self._resolved[ip] = addr._replace(
                hostname=r[0],
                aliaslist=r[1],
                ipaddrlist=r[2],
                updated_at=dt.datetime.utcnow()
            ) if addr else self.Addr(
                hostname=r[0],
                aliaslist=r[1],
                ipaddrlist=r[2],
                updated_at=dt.datetime.utcnow(),
                used_at=dt.datetime.utcnow()
            )
