import numpy as np


class Agg(object):
    """
    Arrgegation function
    """
    factory = None

    def __init__(self, func, name=None):
        self.name = self.__name__ = name or func.__name__
        self._func = func

    def __call__(self, *args, **kwargs):
        return self._func(*args, **kwargs)

    def __str__(self):
        return self.__name__


class AggregatorFactory(object):

    def __init__(self):
        self._inst = {}
        self._factories = {}

    def __iadd__(self, other):
        name = other.__name__
        if name in self._inst:
            raise ValueError('Already exists')

        self._inst[name] = other
        return self

    def __getitem__(self, item):
        return self._inst[item]

    def build(self, prefix, value):
        factory = self._factories[prefix]
        func = factory(value)
        agg = Agg(func, '{}{}'.format(prefix, value))
        self.__iadd__(agg)

    @property
    def factories(self):
        return self._factories

    def add_factory(self, prefix, factory):
        if prefix in self._inst:
            raise ValueError('Already exists')

        self._factories[prefix] = factory


def percentile(n):
    if not 0 < n < 100:
        raise ValueError('n must be > 0 and < 100')

    return lambda a: np.percentile(a, n)


def quartile(n):
    if n not in [1, 2, 3]:
        raise ValueError('n must be in [1, 2, 3]')
    return percentile(n * 25)


aggregators = AggregatorFactory()

aggregators.add_factory('p', percentile)
aggregators.add_factory('q', quartile)

aggregators += Agg(len, 'cnt')
aggregators += Agg(np.sum, 'sum')
aggregators += Agg(np.max, 'max')
aggregators += Agg(np.min, 'min')
aggregators += Agg(np.mean, 'mean')
aggregators.build('p', 95)
aggregators.build('q', 3)
