import logging
import datetime as dt

import stats
import refueller

import urwid
import pandas as pd
import numpy as np


logger = logging.getLogger(__name__)


class TableLine(urwid.WidgetWrap):
    def __init__(self, key, data, title=None):
        title = title or key
        self.id = key
        self.data = data

        w = self.layout(
            urwid.AttrWrap(urwid.Text(title), 'body', 'focus'),
            [urwid.AttrWrap(urwid.Text(d), 'body', 'focus') for d in data]
        )
        super(TableLine, self).__init__(w)

    @classmethod
    def layout(cls, key, cols):
        items = [
            ('fixed', 45, urwid.Padding(key, left=2))
        ]
        items.extend(cols)

        return urwid.Columns(items)

    def selectable(self):
        return True

    def keypress(self, size, key):
        return key


class TableView(urwid.WidgetWrap):
    header_text = ''

    pivot_values = None
    pivot_index = None
    pivot_aggfunc = None

    pivot_sortkey = None
    pivot_ascending = True

    LineCls = TableLine

    def __init__(self):
        self.lbv = urwid.SimpleFocusListWalker([])
        listbox = urwid.ListBox(self.lbv)
        listbox.offset_rows = 1

        self.aggs = [stats.aggregators[name] for name in self.pivot_aggfunc]

        self.frame = urwid.Frame(
            urwid.AttrWrap(listbox, 'body'),
            header=urwid.Text(''),
            footer=urwid.Text(''),
        )
        self.updated_at = None

        super(TableView, self).__init__(self.frame)

    def _create_line(self, key, data, title=None):
        return self.LineCls(key, ['{:0.1f}'.format(data[agg]) for agg in self.pivot_aggfunc], title)

    def _create_header(self):
        cols = []
        for agg in self.aggs:
            if agg.name == self.pivot_sortkey:
                title = u'{} {}'.format(u'\u25B2' if self.pivot_ascending else u'\u25BC', agg.name)
            else:
                title = agg.name

            cols.append(urwid.Text(title))

        return self.LineCls.layout(
            urwid.Text(self.header_text),
            cols
        )

    def update(self, the_r, keep_focus=False, request_source=None):
        self.updated_at = dt.datetime.utcnow()

        samples = the_r.snapshot()
        df = pd.DataFrame(samples.data, columns=refueller.LogEntry._fields)
        if request_source:
            df = df[df['source'] == request_source]

        focus_pos = None

        if not df.empty:
            pivoted = pd.pivot_table(df, values=self.pivot_values, index=self.pivot_index,
                                     aggfunc=self.aggs)

            if self.pivot_sortkey:
                key2 = pivoted[self.pivot_sortkey].rank(ascending=self.pivot_ascending)
                sorter = np.lexsort((key2,))
                pivoted = pivoted.take(sorter)

            focus_key = None
            focus_w, focus_pos = self.lbv.get_focus()
            if focus_w:
                focus_key = focus_w.id

            lines = [self._create_line(key, pivoted.loc[key]) for key in pivoted.index]
            self.lbv[:] = lines

            new_focus = None
            if keep_focus:
                for new_focus, line in enumerate(lines):
                    if line.id == focus_key:
                        break

            res_pos = (new_focus if new_focus is not None else focus_pos) or 0
            res_pos = min(res_pos, len(self.lbv) - 1)
            res_pos = max(res_pos, 0)
            self.lbv.set_focus(res_pos)
        else:
            self.lbv[:] = []

        req_count = len(df['ts'])
        window_secs = the_r.window_period.total_seconds()
        footer = [
            urwid.Text('{}/{}'.format('-' if focus_pos is None else focus_pos + 1, len(self.lbv))),
        ]
        if the_r.requests == refueller.REQUESTS.LAST:
            footer.extend([
                urwid.Text('RPS: {}'.format('{:.02f}'.format(req_count / window_secs) if window_secs else '-')),
                urwid.Text('[last] window: {}'.format(window_secs))
            ])
        elif the_r.requests == refueller.REQUESTS.INPROGRESS:
            footer.append(
                urwid.Text('[inprogress]')
            )

        self.frame.set_footer(urwid.Columns(footer))
        self.frame.set_header(self._create_header())

    def sort_col(self, num):
        if num == 0:
            self.pivot_sortkey = None
            return True
        elif num < 0 or num > len(self.aggs):
            return

        agg = self.aggs[num - 1]
        if agg.name == self.pivot_sortkey:
            self.pivot_ascending = not self.pivot_ascending
        else:
            self.pivot_sortkey = agg.name

        return True


class RequestsView(TableView):
    pivot_values = 'duration'
    pivot_aggfunc = ['sum', 'cnt', 'p95', 'q3', 'max', 'mean']
    pivot_ascending = False


class UsersView(RequestsView):
    header_text = 'Users'
    pivot_index = ['login']


class MethodsView(RequestsView):
    header_text = 'Methods'
    pivot_index = ['method']


class AddrsView(RequestsView):
    header_text = 'Addrs'
    pivot_index = ['raddr']

    def __init__(self, resolver):
        self.resolver = resolver
        super(AddrsView, self).__init__()

    def _create_line(self, key, data, title=None):
        addr = self.resolver.get(key)
        return super(AddrsView, self)._create_line(key, data, addr and addr.hostname)
