#!/usr/bin/env python
"""
The script will count resources, which have too many copies on the host given and will remove
current host's record if requested.
"""

import os
import logging
import argparse

logger = logging.getLogger('scripts.storage_excessive_redundancy')

import common.log
import common.utils
from common import config
from common import console
import common.types.client as ctc

from yasandbox import manager
from yasandbox.database import mapping


def main():
    parser = argparse.ArgumentParser(
        formatter_class=lambda *args, **kwargs: argparse.ArgumentDefaultsHelpFormatter(*args, width=120, **kwargs),
        description='Sandbox storage resources excessive redundancy removal tool.'
    )
    parser.add_argument('--purge', default=False, action='store_true', help="Perform actual resources removal.")
    parser.add_argument('-v', '--verbose', default=False, action='store_true', help="List found resource IDs.")
    parser.add_argument('-l', '--limit', default=0, type=int, help="Limit resources to be listed on host.")
    parser.add_argument(
        "host",
        metavar="HOST", nargs='?',
        help='Host to be checked. Defaults to local host'
    )
    args = parser.parse_args()

    settings = config.Registry()
    node_id = args.host.split('.')[0] if args.host else settings.this.id
    common.log.setup_log(os.path.basename(__file__) + '.log', 'DEBUG')
    logger.info('STARTED')
    with console.LongOperation('Loading resources metainfo and counting host resources. This can take a while'):
        manager.use_locally()
        storages = [c.hostname for c in manager.client_manager.list() if ctc.Tag.STORAGE in c.tags]
        data = mapping.Resource.storage_excessive_redundancy(node_id, storages, args.limit)

    msg = '{} excessive resources on host totally for {}'.format(
        len(data), common.utils.size2str(sum(r['size'] for r in data) << 10)
    )
    print('Detected ' + msg)

    ids = [r['_id'] for r in data]
    logger.debug('Resources list: %r', ids)
    if args.verbose:
        print('Resources list: {!r}'.format(ids))
    if not args.purge:
        return

    with console.LongOperation('Purging excessive resources'):
        mapping.Resource.objects(
            id__in=ids
        ).update(
            pull__hosts_states__host=node_id
        )
    logger.info('Purged ' + msg)

if __name__ == '__main__':
    main()
