from __future__ import division, print_function, unicode_literals

import logging
import collections
import itertools as it

from sandbox.common import abc as common_abc
from sandbox.common import mds as common_mds
from sandbox.common import tvm as common_tvm
from sandbox.common import auth as common_auth
from sandbox.common import enum as common_enum
from sandbox.common import rest as common_rest
from sandbox.common import config as common_config
from sandbox.common import patterns as common_patterns
from sandbox.common.types import misc as ctm

from . import common as commands_common


class Const(common_enum.Enum):
    D_API_URL = None
    D_TVM_SERVICE_ID = None
    SANDBOX_API_URL = None


ParsedResource = collections.namedtuple("ParsedResource", "abc_name abc_id folder_name folder_id resource_id")


SETTINGS = {
    commands_common.Environment.PRODUCTION: {
        Const.D_API_URL: "https://d-api.yandex-team.ru/api/v1",
        Const.D_TVM_SERVICE_ID: "d-production",
        Const.SANDBOX_API_URL: "https://sandbox.yandex-team.ru/api/v2",
    },
    commands_common.Environment.PRE_PRODUCTION: {
        Const.D_API_URL: "https://d.test.yandex-team.ru/api/v1",
        Const.D_TVM_SERVICE_ID: "d-testing",
        Const.SANDBOX_API_URL: "https://www-sandbox1.n.yandex-team.ru/api/v2",
    }
}

PROVIDER = "sandbox"
LIST_LIMIT = 1000

POOL_TAGS = {
    "GENERIC_LINUX_HDD": "GENERIC & LINUX & HDD",
    "GENERIC_LINUX_SSD": "GENERIC & LINUX & SSD",
    "GENERIC_MACOS": "GENERIC & OSX",
    "GENERIC_WINDOWS": "GENERIC & WINDOWS",
    "BROWSER_LINUX_HDD": "BROWSER & LINUX & HDD",
    "BROWSER_LINUX_SSD": "BROWSER & LINUX & SSD",
    "BROWSER_MACOS": "BROWSER & OSX",
    "BROWSER_WINDOWS": "BROWSER & WINDOWS",
    "YABS_LINUX_HDD": "YABS & LINUX & HDD",
    "YABS_LINUX_SSD": "YABS & LINUX & SSD",
    "MOBILE_MONOREPO_MACOS": "MOBILE_MONOREPO & OSX",
}


def _common_init(args):
    logging.root.setLevel(logging.ERROR)
    config = common_config.Registry()
    if args.environment == commands_common.Environment.PRODUCTION:
        config.common.installation = ctm.Installation.PRODUCTION
    else:
        config.common.installation = ctm.Installation.PRE_PRODUCTION
        config.common.mds.s3.idm.url = "https://s3-idm.mdst.yandex.net"


def _create_abcd_client(args):
    d_tvm_service_id = SETTINGS[args.environment][Const.D_TVM_SERVICE_ID]
    tvm_ticket = common_tvm.TVM().get_service_ticket([d_tvm_service_id])[d_tvm_service_id]
    tvm_auth = common_auth.TVMSession(tvm_ticket)
    return common_rest.Client(SETTINGS[args.environment][Const.D_API_URL], auth=tvm_auth)


def _create_sandbox_client(args):
    return common_rest.Client(SETTINGS[args.environment][Const.SANDBOX_API_URL])


def _sandbox_api_list(path, predicate=lambda item: item.get("abcd_account"), **query):
    total = 1
    offset = 0
    while offset < total:
        query.update(limit=LIST_LIMIT, offset=offset)
        response = path.read(**query)
        total = response["total"]
        items = response["items"]
        for item in items:
            if predicate(item):
                yield item
        offset += len(items)


@common_patterns.singleton
def _provider_id(d_api):
    for item in d_api.providers[:]["items"]:
        if item["key"] == PROVIDER:
            return item["id"]


@common_patterns.singleton
def _resource_keys(d_api, provider_id):
    return {
        resource["id"]: resource["key"]
        for resource in _d_api_list(d_api.providers[provider_id].resources)
    }


@common_patterns.singleton
def _resource_ids(d_api, provider_id):
    return dict(map(reversed, _resource_keys(d_api, provider_id).items()))


def _d_api_list(path):
    page_token = None
    while True:
        query = {}
        if page_token:
            query["pageToken"] = page_token
        response = path.read(**query)
        for item in response["items"]:
            yield item
        page_token = response.get("nextPageToken")
        if not page_token:
            break


def _print_quota(abc_name, folder_name, resource_key, quota_item):
    print(
        "{}:{}:{} : balance={} {}, quota={} {}".format(
            abc_name, folder_name, resource_key,
            quota_item["balance"], quota_item["balanceUnitKey"],
            quota_item["quota"], quota_item["quotaUnitKey"],
        )
    )


def list_negative_balance(args):
    _common_init(args)
    d_api = _create_abcd_client(args)
    provider_id = _provider_id(d_api)
    sandbox_api = _create_sandbox_client(args)
    folder_ids = set()
    for item in it.chain.from_iterable(map(_sandbox_api_list, (sandbox_api.group, sandbox_api.resource.buckets))):
        folder_ids.add(item["abcd_account"]["folder_id"])
    for folder_id in folder_ids:
        folder = d_api.folders[folder_id][:]
        for item in _d_api_list(d_api.folders[folder_id].quotas):
            if item["providerId"] != provider_id:
                continue
            balance = item["balance"]
            if balance < 0:
                abc_name = common_abc.abc_service_name(folder["serviceId"])
                resource_key = _resource_keys(d_api, provider_id)[item["resourceId"]]
                _print_quota(abc_name, folder["displayName"], resource_key, item)


def _parse_resource(d_api, provider_id, resource):
    abc_name, folder_name, resource_key = (resource.split(":", 3) + [None] * 3)[:3]
    if abc_name.isdigit():
        abc_id = abc_name
        abc_name = common_abc.abc_service_name(abc_id)
    else:
        abc_id = common_abc.abc_service_id(abc_name)
    assert abc_id, "ABC service {} not found".format(abc_name)
    if not folder_name:
        folder_name = "default"
    folder = next(
        filter(lambda f: f["displayName"] == folder_name, _d_api_list(d_api.services[abc_id].folders)), None
    )
    assert folder, "folder '{}' for ABC service '{}' not found".format(folder_name, abc_name)
    resource_id = _resource_ids(d_api, provider_id).get(resource_key) if resource_key else None
    assert not resource_key or resource_id, "resource '{}' in folder '{}' of ABC service '{}' not found".format(
        resource_key, folder_name, abc_name
    )
    return ParsedResource(abc_name, abc_id, folder_name, folder["id"], resource_id)


def get_quota(args):
    _common_init(args)
    d_api = _create_abcd_client(args)
    provider_id = _provider_id(d_api)
    parsed_resource = _parse_resource(d_api, provider_id, args.resource)
    resource_keys = _resource_keys(d_api, provider_id)
    for item in _d_api_list(d_api.folders[parsed_resource.folder_id].quotas):
        if item["providerId"] != provider_id:
            continue
        if not parsed_resource.resource_id or item["resourceId"] == parsed_resource.resource_id:
            _print_quota(parsed_resource.abc_name, parsed_resource.folder_name, resource_keys[item["resourceId"]], item)


@common_patterns.singleton
def _allowed_units(d_api, resource_id):
    provider_id = _provider_id(d_api)
    resource = d_api.providers[provider_id].resourcesDirectory.resources[resource_id][:]
    units = []
    for unit_id in resource["allowedUnitIds"]:
        units.append(d_api.unitsEnsembles[resource["unitsEnsembleId"]].units[unit_id][:])
    return units


def _find_min_unit(allowed_units):
    unit_base = None
    min_unit = None
    for unit in allowed_units:
        if unit_base is None:
            unit_base = unit["base"]
        assert unit_base == unit["base"]
        if min_unit is None or min_unit["power"] > unit["power"]:
            min_unit = unit
    assert min_unit is not None
    return min_unit


def _convert(value, unit_src, unit_dst):
    if unit_src["id"] == unit_dst["id"]:
        return value
    assert unit_src["base"] == unit_dst["base"]
    delta_power = unit_src["power"] - unit_dst["power"]
    if delta_power >= 0:
        value *= unit_src["base"]**abs(delta_power)
    else:
        value //= unit_src["base"]**abs(delta_power)
    return value


def _add_quota_request(d_api, args, delta, current_quota, parsed_resource):
    provider_id = _provider_id(d_api)
    allowed_units = {unit["key"]: unit for unit in _allowed_units(d_api, parsed_resource.resource_id)}
    assert args.unit in allowed_units, "unit must be one of {}".format(list(allowed_units.keys()))
    if current_quota is None:
        balance = quota = delta
        balance_unit_key = quota_unit_key = args.unit
    else:
        quota_unit_key = current_quota["quotaUnitKey"]
        quota_unit = allowed_units[quota_unit_key]
        balance_unit_key = current_quota["balanceUnitKey"]
        balance_unit = allowed_units[balance_unit_key]
        delta_unit = allowed_units[args.unit]

        min_unit = _find_min_unit([quota_unit, balance_unit, delta_unit])

        balance = _convert(current_quota["balance"], balance_unit, min_unit)
        quota = _convert(current_quota["quota"], quota_unit, min_unit)
        delta = _convert(delta, delta_unit, min_unit)
        quota += delta
        balance += delta
    return {
        "serviceId": parsed_resource.abc_id,
        "resourceQuotas": [
            {
                "resourceId": parsed_resource.resource_id,
                "providerId": provider_id,
                "quota": quota,
                "quotaUnitKey": quota_unit_key,
                "balance": balance,
                "balanceUnitKey": balance_unit_key,
            },
        ],
    }


def add_quota(args):
    _common_init(args)
    d_api = _create_abcd_client(args)
    provider_id = _provider_id(d_api)
    parsed_resource = _parse_resource(d_api, provider_id, args.resource)
    assert parsed_resource.resource_id, "resource_id is required"
    allowed_units = {unit["key"] for unit in _allowed_units(d_api, parsed_resource.resource_id)}
    assert args.unit in allowed_units, "unit must be one of {}".format(allowed_units)
    current_quota = None
    for item in _d_api_list(d_api.folders[parsed_resource.folder_id].quotas):
        if item["resourceId"] == parsed_resource.resource_id:
            current_quota = item
            break
    quotas = [_add_quota_request(d_api, args, args.quota, current_quota, parsed_resource)]
    print(d_api["import"]._importQuotas({"quotas": quotas}))


def move_quota(args):
    _common_init(args)
    d_api = _create_abcd_client(args)
    provider_id = _provider_id(d_api)
    parsed_src_resource = _parse_resource(d_api, provider_id, args.src_resource)
    parsed_dst_resource = _parse_resource(d_api, provider_id, args.dst_resource)
    assert parsed_src_resource.resource_id, "src resource_id is required"
    assert parsed_dst_resource.resource_id, "dst resource_id is required"
    assert parsed_src_resource.resource_id == parsed_dst_resource.resource_id, "src anc dst resource_id must be equal"
    allowed_units = {unit["key"] for unit in _allowed_units(d_api, parsed_src_resource.resource_id)}
    assert args.unit in allowed_units, "unit must be one of {}".format(allowed_units)
    quotas = []
    for parsed_resource, delta in ((parsed_src_resource, -args.quota), (parsed_dst_resource, args.quota)):
        current_quota = None
        for item in _d_api_list(d_api.folders[parsed_resource.folder_id].quotas):
            if item["resourceId"] == parsed_resource.resource_id:
                current_quota = item
                break
        quotas.append(_add_quota_request(d_api, args, delta, current_quota, parsed_resource))
    print(d_api["import"]._importQuotas({"quotas": quotas}))


def sandbox_summary(args):
    _common_init(args)
    d_api = _create_abcd_client(args)
    sandbox_api = _create_sandbox_client(args)
    provider_id = _provider_id(d_api)
    resource_ids = _resource_ids(d_api, provider_id)
    resource_key_column_length = max(len(resource_key) for resource_key in resource_ids) + 2
    for resource_key in sorted(resource_ids):
        if args.resource_key and resource_key != args.resource_key:
            continue
        tags = POOL_TAGS.get(resource_key)
        if tags is None:
            total_unit = "gibibytes"
            total = sum(
                common_mds.S3.bucket_stats(item["name"])["max_size"] >> 30
                for item in _sandbox_api_list(sandbox_api.resource.buckets)
            )
        else:
            total_unit = "cores"
            total = sum(
                item["ncpu"]
                for item in _sandbox_api_list(sandbox_api.client, predicate=lambda _: True, tags=tags, fields="ncpu")
            )
        print("{{:<{}s}}: {{}} {{}}".format(resource_key_column_length).format(resource_key, total, total_unit))


def abcd_summary(args):
    _common_init(args)
    d_api = _create_abcd_client(args)
    sandbox_api = _create_sandbox_client(args)
    provider_id = _provider_id(d_api)
    resource_ids = _resource_ids(d_api, provider_id)
    if args.full_scan:
        folder_ids = [folder["id"] for folder in _d_api_list(d_api.folders)]
    else:
        folder_ids = set()
        for item in it.chain.from_iterable(map(_sandbox_api_list, (sandbox_api.group, sandbox_api.resource.buckets))):
            folder_ids.add(item["abcd_account"]["folder_id"])
    resource_key_column_length = max(len(resource_key) for resource_key in resource_ids) + 2
    for resource_key, resource_id in sorted(resource_ids.items()):
        if args.resource_key and resource_key != args.resource_key:
            continue
        allowed_units = {unit["key"]: unit for unit in _allowed_units(d_api, resource_id)}
        total = 0
        total_unit = None
        for folder_id in folder_ids:
            try:
                quota = d_api.folders[folder_id].providers[provider_id].resources[resource_id].quota[:]
                if total_unit is None:
                    total = quota["quota"]
                    total_unit = allowed_units[quota["quotaUnitKey"]]
                else:
                    quota_unit = allowed_units[quota["quotaUnitKey"]]
                    new_total_unit = _find_min_unit([total_unit, quota_unit])
                    total = _convert(total, total_unit, new_total_unit)
                    total += _convert(quota["quota"], quota_unit, new_total_unit)
                    total_unit = new_total_unit
            except common_rest.Client.HTTPError as ex:
                if ex.status != common_rest.requests.codes.NOT_FOUND:
                    raise
        total_unit_key = total_unit["key"] if total_unit else ""
        if total_unit and total_unit["key"] == "millicores":
            total_unit_key = "cores"
            old_total_unit, total_unit = total_unit, allowed_units[total_unit_key]
            total = _convert(total, old_total_unit, total_unit)
        print("{{:<{}s}}: {{}} {{}}".format(resource_key_column_length).format(resource_key, total, total_unit_key))


def setup_negative_balance(parser):
    parser.set_defaults(func=list_negative_balance)


def setup_get_quota(parser):
    parser.add_argument(
        "-r", "--resource",
        metavar="<resource>", type=str, required=True,
        help="resource description: abc_name[:[folder_key][:resource_key]]",
    )
    parser.set_defaults(func=get_quota)


def setup_add_quota(parser):
    parser.add_argument(
        "-r", "--resource",
        metavar="<resource>", type=str, required=True,
        help="resource description: abc_name:[folder_key]:resource_key",
    )
    parser.add_argument(
        "-q", "--quota",
        metavar="<quota>", type=int, required=True,
        help="quota to add (can be negative, to subtract)",
    )
    parser.add_argument(
        "-u", "--unit",
        metavar="<unit>", type=str, required=True,
        help="quota unit",
    )
    parser.set_defaults(func=add_quota)


def setup_move_quota(parser):
    parser.add_argument(
        "-s", "--src-resource",
        metavar="<src_resource>", type=str, required=True,
        help="source resource description: abc_name:[folder_key]:resource_key",
    )
    parser.add_argument(
        "-d", "--dst-resource",
        metavar="<dst_resource>", type=str, required=True,
        help="destination resource description: abc_name:[folder_key]:resource_key",
    )
    parser.add_argument(
        "-q", "--quota",
        metavar="<quota>", type=int, required=True,
        help="quota to move",
    )
    parser.add_argument(
        "-u", "--unit",
        metavar="<unit>", type=str, required=True,
        help="quota unit",
    )
    parser.set_defaults(func=move_quota)


def setup_sandbox_summary(parser):
    parser.add_argument(
        "-r", "--resource-key",
        metavar="<resource_key>", type=str, required=False,
        help="resource key",
    )
    parser.set_defaults(func=sandbox_summary)


def setup_abcd_summary(parser):
    parser.add_argument(
        "-r", "--resource-key",
        metavar="<resource_key>", type=str, required=False,
        help="resource key",
    )
    parser.add_argument(
        "-f", "--full-scan",
        action="store_true",
        help="full scan ABCD folders",
    )
    parser.set_defaults(func=abcd_summary)


def setup_parser(parser):
    parser.add_argument(
        "-e", "--environment", metavar="<environment>", type=str, help="<environment>",
        choices=list(commands_common.Environment), default=commands_common.Environment.PRODUCTION
    )
    subparsers = parser.add_subparsers(help="sub-command help")

    negative_balance_parser = subparsers.add_parser("negative-balance", help="get folders with negative balance")
    setup_negative_balance(negative_balance_parser)

    get_quota_parser = subparsers.add_parser("get-quota", help="get quota(s) for ABC service")
    setup_get_quota(get_quota_parser)

    add_quota_parser = subparsers.add_parser("add-quota", help="add quota for ABC service")
    setup_add_quota(add_quota_parser)

    move_quota_parser = subparsers.add_parser("move-quota", help="move quota between ABC services")
    setup_move_quota(move_quota_parser)

    sandbox_summary_parser = subparsers.add_parser(
        "sandbox-summary", help="total resource quantity according to the Sandbox"
    )
    setup_sandbox_summary(sandbox_summary_parser)

    abcd_summary_parser = subparsers.add_parser(
        "abcd-summary", help="total resource quantity according to the ABCD"
    )
    setup_abcd_summary(abcd_summary_parser)
