# coding: utf-8
from __future__ import division, print_function, unicode_literals

import collections
import functools
import hashlib
import json

import six
from six.moves import input, urllib
from distutils.util import strtobool

from .. import consts, common_args, utils


def dict_hash(d):
    return hashlib.sha1(json.dumps(d, sort_keys=True).encode("utf-8")).hexdigest()


class Components(object):
    PARAMS = {
        "client": "create_client_resource",
        "fileserver": "build_fileserver",
        "agentr": "create_agentr_resource",
        "server": "create_server_resource",
        "serviceapi": "create_serviceapi_resource",
        "taskbox": "create_taskbox_resource",
        "serviceq": "create_serviceq_resource",
        "services": "create_services_resource",
        "proxy": "create_proxy_resource",
        "library": "create_library_resource",
        "ui": "create_ui_resource",
    }

    @classmethod
    def to_params(cls, complist):
        complist = set(complist)
        return {param: alias in complist for alias, param in six.iteritems(cls.PARAMS)}


def get_params(env, components, arcadia_patch):
    params = collections.OrderedDict(
        [
            ("build_type", env),
            ("create_samogon_resource", True),
            ("skip_tests", True),
            ("sandbox_branch_name", "trunk"),
            ("precompile", True),
            ("patch_from_rb", bool(arcadia_patch)),
            ("arcadia_patch", arcadia_patch),
        ]
    )
    params.update(Components.to_params(components))
    return params


def build_sandbox(api_args, task_args, builds, components):

    print("You are going to build: {}".format(", ".join(components) or "<none>"))
    print("For the following environments: {}".format(", ".join(builds) or "<none>"))

    if task_args.arcadia_patch:
        print("With patch: {}".format(task_args.arcadia_patch))

    while True:
        answer = input("Continue? [Y/n] ").strip() or "yes"
        try:
            if strtobool(answer):
                break
            else:
                return
        except Exception as exc:
            print(exc)
            pass

    parsed_base_url = urllib.parse.urlparse(api_args.base_url)
    client = utils.sandbox_client(api_args.base_url, auth=api_args.auth)

    task_ids = {}

    for env in builds:
        input_params = get_params(env, components, task_args.arcadia_patch)
        description = "{} Sandbox{}".format(
            env.upper(),
            " (with patch {})".format(task_args.arcadia_patch) if task_args.arcadia_patch else ""
        )
        data = dict(
            type="BUILD_SANDBOX",
            owner=task_args.owner,
            kill_timeout=60 * 60,
            custom_fields=[{"name": name, "value": value} for name, value in six.iteritems(input_params)],
            tags=[env.upper()],
            priority={"class": "SERVICE", "subclass": "HIGH"},
            notifications=[],
            description=description,
        )
        data["uniqueness"] = {"excluded_statuses": list(consts.UNIQUE_EXCLUDED_STATUSES), "key": dict_hash(data)}

        task_ids[env] = client.task(**data)["id"]

    if task_args.draft:
        print(task_ids)
        return

    resp = client.batch.tasks.start.update(id=list(task_ids.values()))

    for result in resp:
        url = parsed_base_url._replace(path="/task/{}".format(result["id"]))
        url = urllib.parse.urlunparse(url)
        status = result["status"]

        print("%s: %s" % (status, url))
        if status != "SUCCESS":
            print("    {}".format(result["message"]))


def main(args):
    api_args = common_args.API.create(args)
    task_args = common_args.Task.create(args)
    build_sandbox(api_args=api_args, task_args=task_args, builds=args.builds, components=args.components)


def setup_parser(parser):

    group = parser.add_argument_group("Build Parameters")

    validate = functools.partial(utils.list_parse, allowed={"prestable", "stable"})

    default_builds = ["prestable", "stable"]
    group.add_argument(
        "-b", "--builds", default=["prestable", "stable"], type=validate,
        help="List of build types, default: `{}`".format(",".join(default_builds))
    )

    validate = functools.partial(utils.list_parse, allowed=set(Components.PARAMS))

    cmpts = group.add_mutually_exclusive_group(required=True)
    cmpts.add_argument(
        "-c", dest="components", default=[], type=validate,
        help="List of components to build, valid items: {}".format(", ".join(('`%s`' % p for p in Components.PARAMS)))
    )
    cmpts.add_argument(
        "--all", dest='components', action='store_const', const=tuple(set(Components.PARAMS) - {"ui"}),
        help="Build all components (except UI)"
    )

    common_args.API.setup(parser)
    common_args.Task.setup(parser)

    parser.set_defaults(func=main)
