# coding: utf-8
from __future__ import division, print_function, unicode_literals

import io
from datetime import datetime as dt

from dateutil import parser as dt_parser
from library import format as sky_format
import terminaltables

from .. import common_args, consts, utils


class Formatter(object):

    def __init__(self, dead):
        self.dead = dead

    @property
    def skynet(self):
        return sky_format.formatHosts([h["fqdn"] for h in self.dead], useGrouping=True, addDomain=True)

    @property
    def table(self):
        output = io.StringIO()

        dead = sorted(self.dead, key=lambda x: dt_parser.parse(x["last_activity"]), reverse=True)

        lines = [("Client", "FQDN", "Last Ping")]
        for client in dead:
            td = utils.duration_string(dt.utcnow() - dt_parser.parse(client["last_activity"], ignoretz=True))
            lines.append((client["id"], client["fqdn"], td))

        table = terminaltables.SingleTable(lines)

        output.write(table.table)
        output.write("\n")
        output.write("Total: {}\n".format(len(dead)))
        return output.getvalue()


def print_dead_clients(api, tag_query, fmt):
    client = utils.sandbox_client(api.base_url, auth=api.auth)
    data = client.client.read(
        alive=0, offset=0, limit=consts.DEAD_CLIENTS_LIMIT, tags=str(tag_query or ""), fields="id,last_activity,fqdn"
    )
    print(getattr(Formatter(data["items"]), fmt))


def main(args):
    api = common_args.API.create(args)
    clients_filter = common_args.ClientsFilter.create(args)
    print_dead_clients(api=api, tag_query=clients_filter.tag_query, fmt=args.format)


def setup_parser(parser):
    parser.add_argument("-f", "--format", choices=("skynet", "table"), default="table", help="Output format")
    common_args.ClientsFilter.setup(parser)
    common_args.API.setup(parser)
    parser.set_defaults(func=main)
