import json
import argparse
import operator as op
import itertools as it
import dataclasses

from typing import List, Optional, Any
from distutils.util import strtobool

from sandbox.common import rest as common_rest
from sandbox.common import patterns as common_patterns

from .. import utils

# API WIKI: https://wiki.yandex-team.ru/intranet/abc/dispenser/docs/api
# Swagger: https://dispenser.yandex-team.ru/common/api/api-docs?url=/api/swagger.json#
# OAuth token: https://oauth.yandex-team.ru/authorize?response_type=token&client_id=fe21371385fa4ac8a24c2e9c318e0561


class Dispenser:
    DISPENSER_API_URL = "https://dispenser.yandex-team.ru/common/api/v1"

    @common_patterns.singleton_classproperty
    def client(cls) -> common_rest.Client:
        return common_rest.Client(cls.DISPENSER_API_URL, auth=utils.oauth_token())


@dataclasses.dataclass
class ResourceChange:
    name: str
    key: str
    service: str
    amount: int
    unit: str
    amount_str: str
    order_id: int
    order_date: str
    segments: List[str]
    allocated: bool

    def __str__(self):
        return "{resource}: {amount} [{segments}] on {order} ({state})".format(
            resource=self.name, amount=self.amount_str, segments=','.join(self.segments),
            order=self.order_date, state="completed" if self.allocated else "not allocated"
        )

    @classmethod
    def from_json(cls, data: dict) -> "ResourceChange":
        return cls(
            name=data["resource"]["name"],
            key=data["resource"]["key"],
            service=data["service"]["key"],
            amount=data["amount"]["value"],
            unit=data["amount"]["unit"],
            amount_str="{} {}".format(
                data["amount"]["humanized"]["stringValue"], data["amount"]["humanized"]["abbreviation"]
            ),
            order_id=data["order"]["id"],
            order_date=data["order"]["orderDate"],
            segments=data["segmentKeys"],
            allocated=data["amount"]["value"] == data["amountAllocated"]["value"]
        )


@dataclasses.dataclass
class ResourcesRequest:
    request_id: int
    ticket: str
    service_id: int
    service_name: str
    actual_changes: List[ResourceChange]

    def __str__(self):
        return "Request #{} for service '{}' (abc: {}): https://st.yandex-team.ru/{}\n{}".format(
            self.request_id, self.service_name, self.service_id, self.ticket,
            '\n'.join(map(str, self.actual_changes))
        )

    @classmethod
    def from_json(cls, data: dict, provider: str, orders: List[int], remained_only: bool) -> "ResourcesRequest":
        changes = map(ResourceChange.from_json, data["changes"])
        return cls(
            request_id=data["id"],
            ticket=data["trackerIssueKey"],
            service_id=data["project"]["abcServiceId"],
            service_name=data["project"]["name"],
            actual_changes=[
                change for change in changes
                if change.order_id in orders and not (remained_only and change.allocated) and change.service == provider
            ]
        )

    def allocate(self, comment: str) -> None:
        allocate_request = {
            "updates": [{
                "requestId": self.request_id,
                "comment": comment,
                "changes": [
                    {
                        "resourceKey": change.key,
                        "serviceKey": change.service,
                        "bigOrderId": change.order_id,
                        "segmentKeys": change.segments,
                        "amountAllocated": {
                            "value": change.amount,
                            "unit": change.unit
                        }
                    } for change in self.actual_changes
                ]
            }]
        }
        print("Resulted request:\n{}".format(json.dumps(allocate_request, indent=4)))
        if ask_for_approve("Send allocation request? [y/n]: "):
            print(Dispenser.client["resource-preorder/quotaState"].patch())


def list_orders(_: Any) -> None:
    big_orders = Dispenser.client.bot["big-orders"].read()["result"]
    print("Date  OrderId")
    sorted_orders = sorted(filter(op.itemgetter("valid"), big_orders), key=op.itemgetter("orderDate"))
    for date, group in it.groupby(sorted_orders, key=op.itemgetter("orderDate")):
        print("{}  {}".format(date, ",".join(map(str, map(op.itemgetter("id"), group)))))


def _get_resources_requests(
    provider: str, orders: List[int], remained_only: bool = True,
    required_requests: Optional[List[int]] = None, required_tickets: Optional[List[str]] = None,
) -> List[ResourcesRequest]:
    resource_requests = Dispenser.client["quota-requests"].read(
        params={"service": provider, "status": "CONFIRMED"}
    )["result"]
    parsed_requests = [
        ResourcesRequest.from_json(request_data, provider, orders, remained_only) for request_data in resource_requests
    ]
    parsed_requests = [request for request in parsed_requests if request.actual_changes]
    if required_requests:
        return [request for request in parsed_requests if request.request_id in required_requests]
    elif required_tickets:
        return [request for request in parsed_requests if request.ticket in required_tickets]
    else:
        return parsed_requests


def list_requests(args) -> None:
    resource_requests = _get_resources_requests(args.service, args.orders, remained_only=args.remained)
    print("\n\n".join(map(str, resource_requests)))


def allocate_resources(args) -> None:
    resource_requests = _get_resources_requests(
        args.service, args.orders, required_requests=args.requests, required_tickets=args.tickets
    )
    for resource_request in resource_requests:
        print(resource_request)
        if ask_for_approve("Allocate these resources? [y/n]: "):
            resource_request.allocate(args.comment)


def ask_for_approve(message: str) -> bool:
    while True:
        try:
            return strtobool(input(message))
        except ValueError:
            print("Wrong input. ", end="")


def setup_list_orders(parser: argparse.ArgumentParser):
    parser.set_defaults(func=list_orders)


def setup_list_requests(parser: argparse.ArgumentParser):
    parser.add_argument("-s", "--service", default="sandbox", help="filter requests by provider service")
    parser.add_argument(
        "-o", "--orders", metavar="ORDER_ID", nargs="+", type=int, required=True,
        help="filter requests by big order id"
    )
    parser.add_argument("--remained", action="store_true", help="show only incomplete requests")
    parser.set_defaults(func=list_requests)


def setup_allocate_resources(parser: argparse.ArgumentParser):
    parser.add_argument(
        "-s", "--service", default="sandbox",
        help="filter requests by provider service (default: sandbox)"
    )
    parser.add_argument(
        "-o", "--orders", metavar="ORDER_ID", nargs="+", type=int, required=True,
        help="filter requests by big order id"
    )
    selector = parser.add_mutually_exclusive_group()
    selector.add_argument(
        "-r", "--requests", metavar="REQUEST_ID", nargs="+", type=int,
        help="select specific requests by id"
    )
    selector.add_argument(
        "-t", "--tickets", metavar="ORDER_ID", nargs="+",
        help="select specific requests by its ticket"
    )
    parser.add_argument(
        "-c", "--comment", required=True,
        help="commentary for requests update (e.g. order acceptance ticket)"
    )
    parser.set_defaults(func=allocate_resources)


def setup_parser(parser: argparse.ArgumentParser):
    subparsers = parser.add_subparsers(help="sub-command help")

    list_orders_parser = subparsers.add_parser("list-orders", help="list actual big orders")
    setup_list_orders(list_orders_parser)

    list_requests_parser = subparsers.add_parser("list-requests", help="list confirmed resources requests")
    setup_list_requests(list_requests_parser)

    allocate_resources_parser = subparsers.add_parser("allocate-resources", help="mark resources as allocated")
    setup_allocate_resources(allocate_resources_parser)
