from __future__ import division, print_function, unicode_literals

import json
import requests

import six

from library.python import oauth
from library.python.vault_client.instances import Production as VaultClient

import boto3

from sandbox.common import auth as common_auth
from sandbox.common import enum as common_enum
from sandbox.common import rest as common_rest
from sandbox.common import config as common_config
from sandbox.common import format as common_format
from sandbox.common import patterns as common_patterns

from . import common as commands_common

from .. import utils


class Constants(common_enum.Enum):
    common_enum.Enum.lower_case()

    S3_URL_MANAGE = None
    S3_URL = None
    YAV_KEY = None
    SANDBOX_API_URL = None


SETTINGS = {
    commands_common.Environment.PRODUCTION: {
        Constants.S3_URL_MANAGE: "https://s3-idm.mds.yandex.net",
        Constants.S3_URL: "http://s3.mds.yandex.net",
        Constants.YAV_KEY: "production_mds_s3_credentials",
        Constants.SANDBOX_API_URL: "https://sandbox.yandex-team.ru/api/v2",
    },
    commands_common.Environment.PRE_PRODUCTION: {
        Constants.S3_URL_MANAGE: "https://s3-idm.mdst.yandex.net",
        Constants.S3_URL: "http://s3.mdst.yandex.net",
        Constants.YAV_KEY: "preproduction_mds_s3_credentials",
        Constants.SANDBOX_API_URL: "https://www-sandbox1.n.yandex-team.ru/api/v2",
    }
}


class S3(object):

    YAV_SECRET_ID = "sec-01cxzb7ycnmdqfyvrpgt8xqjme"

    @common_patterns.singleton_classproperty
    def oauth_token(cls):
        return utils.oauth_token()

    @common_patterns.singleton_classproperty
    def manager_headers(cls):
        headers = dict(common_auth.OAuth(S3.oauth_token))
        headers.update({
            "Content-Type": "application/json"
        })
        return headers

    @classmethod
    @common_patterns.singleton
    def s3_keys(cls, environment):
        client = VaultClient(decode_files=True)

        keys_settings = client.get_version(S3.YAV_SECRET_ID)["value"][SETTINGS[environment][Constants.YAV_KEY]]

        access_key_id = keys_settings.split("\n")[1].split(" = ")[1]
        secret_access_key = keys_settings.split("\n")[2].split(" = ")[1]
        return access_key_id, secret_access_key

    @classmethod
    @common_patterns.singleton
    def boto_client(cls, environment):
        aws_access_key_id, aws_secret_access_key = cls.s3_keys(environment)
        session = boto3.session.Session(
            aws_access_key_id=aws_access_key_id,
            aws_secret_access_key=aws_secret_access_key,
        )
        return session.client(
            service_name="s3",
            endpoint_url=SETTINGS[environment][Constants.S3_URL]
        )


def get_state(args):
    client = common_rest.Client(
        SETTINGS[args.environment][Constants.S3_URL_MANAGE],
        auth=S3.oauth_token,
        max_timeout=30,
    )
    info = client.stats.buckets[args.bucket].read()
    print(json.dumps(info, indent=4))


def set_quota(args):
    r = requests.patch(
        "/".join((SETTINGS[args.environment][Constants.S3_URL_MANAGE], "management/bucket", args.bucket)),
        headers=S3.manager_headers,
        data=json.dumps({"max_size": common_format.str2size(str(args.max_size) + str(args.measure))})
    )
    if r.status_code >= 400:
        print(r.text)
    r.raise_for_status()
    print("Quota changed")
    print(json.dumps(r.json(), indent=4))


def get_policies(args):
    boto_client = S3.boto_client(args.environment)
    policies = boto_client.get_bucket_lifecycle_configuration(Bucket=args.bucket)
    print(json.dumps(policies, indent=4))


def set_policies(args):
    boto_client = S3.boto_client(args.environment)
    _set_bucket_policies(boto_client, args.bucket)


def _set_bucket_policies(boto_client, bucket):
    print("Setting bucket policies")
    boto_client.put_bucket_lifecycle_configuration(
        Bucket=bucket,
        LifecycleConfiguration={
            "Rules": [
                {
                    "ID": "Abort3DaysMultiparts",
                    "Filter": {
                        "Prefix": ""
                    },
                    "Status": "Enabled",
                    "AbortIncompleteMultipartUpload": {
                        "DaysAfterInitiation": 3
                    }
                }
            ]
        }
    )
    print(boto_client.get_bucket_lifecycle_configuration(Bucket=bucket))


def _create_bucket_object_in_sandbox(args):
    print("Creating bucket object in Sandbox")
    auth_config = common_config.Registry().server.auth.oauth.apps.api
    token = oauth.get_token(client_id=auth_config.client_id, client_secret=auth_config.client_secret)
    sandbox_api = common_rest.Client(
        SETTINGS[args.environment][Constants.SANDBOX_API_URL],
        auth=common_auth.OAuth(token),
    )
    sandbox_api.resource.buckets(name=args.bucket)


def create_bucket(args):
    boto_client = S3.boto_client(args.environment)
    result = boto_client.create_bucket(Bucket=args.bucket)
    print(json.dumps(result, indent=4))
    data = {"is_anonymous_read_enabled": True}
    if args.max_size is not None:
        data["max_size"] = common_format.str2size(str(args.max_size) + str(args.measure))
    r = requests.patch(
        "/".join((SETTINGS[args.environment][Constants.S3_URL_MANAGE], "management/bucket", args.bucket)),
        headers=S3.manager_headers,
        data=json.dumps(data)
    )
    if r.status_code >= 400:
        print(r.text)
    r.raise_for_status()
    print("Anonymous read{} changed".format(" and quota" if args.max_size is not None else ""))
    print(json.dumps(r.json(), indent=4))
    _set_bucket_policies(boto_client, args.bucket)
    _create_bucket_object_in_sandbox(args)


def list_buckets(args):
    boto_client = S3.boto_client(args.environment)
    buckets = boto_client.list_buckets()["Buckets"]
    result_bucket = None
    for bucket in buckets:
        for k in six.iterkeys(bucket):
            bucket[k] = str(bucket[k])
        if args.bucket is not None and bucket["Name"] == args.bucket:
            result_bucket = bucket
            break
    if args.bucket is not None and result_bucket is None:
        print("Bucket {} not found".format(args.bucket))
    elif result_bucket:
        print(json.dumps(result_bucket, indent=4))
    else:
        print(json.dumps(buckets, indent=4))


def setup_get_state_parser(parser):
    parser.add_argument(
        "-b", "--bucket",
        metavar="<bucket>", type=str, required=True,
        help="S3 bucket name",
    )
    parser.set_defaults(func=get_state)


def _setup_bucket_size_args(parser):
    parser.add_argument(
        "-b", "--bucket",
        metavar="<bucket>", type=str, required=True,
        help="S3 bucket name",
    )
    parser.add_argument(
        "-s", "--max-size",
        metavar="<max_size>", type=int, required=True,
        help="Bucket size",
    )
    parser.add_argument(
        "-m", "--measure",
        metavar="<measure>", type=str, required=True, choices=common_format.SIZE_SUFFIXES, default="B",
        help="Measure for bucket size ({})".format("/".join(common_format.SIZE_SUFFIXES)),
    )


def setup_set_quota_parser(parser):
    _setup_bucket_size_args(parser)
    parser.set_defaults(func=set_quota)


def setup_create_bucket(parser):
    _setup_bucket_size_args(parser)
    parser.set_defaults(func=create_bucket)


def setup_list_buckets(parser):
    parser.add_argument("-b", "--bucket", metavar="<bucket>", type=str, help="S3 bucket name", default=None)
    parser.set_defaults(func=list_buckets)


def setup_get_policies_parser(parser):
    parser.add_argument("-b", "--bucket", metavar="<bucket>", type=str, help="S3 bucket name", default=None)
    parser.set_defaults(func=get_policies)


def setup_set_policies_parser(parser):
    parser.add_argument("-b", "--bucket", metavar="<bucket>", type=str, help="S3 bucket name", default=None)
    parser.set_defaults(func=set_policies)


def setup_parser(parser):
    parser.add_argument(
        "-e", "--environment", metavar="<environment>", type=str, help="<environment>",
        choices=list(commands_common.Environment), default=commands_common.Environment.PRODUCTION
    )
    subparsers = parser.add_subparsers(help="sub-command help")

    get_state_parser = subparsers.add_parser("state", help="get bucket state")
    setup_get_state_parser(get_state_parser)

    set_quota_parser = subparsers.add_parser("set-quota", help="set bucket quota")
    setup_set_quota_parser(set_quota_parser)

    create_bucket_parser = subparsers.add_parser("create-bucket", help="create bucket")
    setup_create_bucket(create_bucket_parser)

    list_buckets_parser = subparsers.add_parser("list-buckets", help="list buckets")
    setup_list_buckets(list_buckets_parser)

    get_policies_parser = subparsers.add_parser("get-policies", help="get bucket policies")
    setup_get_policies_parser(get_policies_parser)

    set_policies_parser = subparsers.add_parser("set-policies", help="set bucket policies")
    setup_set_policies_parser(set_policies_parser)
