# coding: utf-8
import sys
import terminaltables
import time
import itertools

from sandbox import common
from concurrent import futures
from library.python import oauth
from sandbox.common import console
from sandbox.common.patterns import singleton_property

from .. import consts
from .. import host_classes as hc


class Conductor:
    def __init__(self):
        token = oauth.get_token(client_id=consts.OAUTH_CLIENT["id"], client_secret=consts.OAUTH_CLIENT["secret"])
        self.client = common.rest.Client("https://c.yandex-team.ru/api/v2", auth=token)
        self._resolved_groups = dict()

    def _get_conductor_hosts_by_group(self, group_id, bar):
        hosts = set()

        first_query, page_number, data = True, 1, None

        while data or first_query:
            if not first_query and data:
                hosts.update(
                    hc.ConductorHost(
                        host["attributes"]["fqdn"],
                        host["attributes"]["inventory_number"],
                        group_id
                    ) for host in data
                )

            first_query = False

            data = self.client.groups[group_id].hosts.read(
                {"page[number]": str(page_number), "page[size]": "20"}
            ).get("data")

            page_number += 1
        bar.add(1)

        return hosts

    def _get_conductor_descendants_groups(self, root_groups):
        bar = console.ProgressBar("Getting descendant groups", len(root_groups))
        res = set()

        for root in root_groups:
            group = self.client.groups[root].relationships.descendants.read().get("data")
            for subgroup in group:
                res.add(subgroup["id"])
            bar.add(1)
        res.update(root_groups)
        bar.finish()

        return res

    def _get_conductor_hosts_in_parallel(self, group_ids, bar):
        with futures.ThreadPoolExecutor(max_workers=32) as executor:
            hosts = [
                executor.submit(self._get_conductor_hosts_by_group, group_id, bar) for group_id in group_ids
            ]
            return set(itertools.chain.from_iterable(host.result() for host in hosts))

    def _get_conductor_hosts_by_root_groups(self, root_groups):
        ids = self._get_conductor_descendants_groups(root_groups)
        bar = console.ProgressBar(
            "Getting conductor hosts by root groups at {}".format(self.client.host),
            len(ids)
        )
        hosts = self._get_conductor_hosts_in_parallel(ids, bar)
        bar.finish()
        return hosts

    @singleton_property
    def hosts(self):
        return self._get_conductor_hosts_by_root_groups(consts.ROOT_GROUPS)

    @singleton_property
    def win_hosts(self):
        return self._get_conductor_hosts_by_root_groups(consts.WIN_GROUP)

    def resolve_group_id(self, group_id):
        if not group_id.isnumeric() or not group_id:
            return group_id
        if group_id not in self._resolved_groups:
            self._resolved_groups[group_id] = self.client.groups[group_id].read()["data"]["attributes"]["name"]
        return self._resolved_groups[group_id]


class Walle:
    def __init__(self):
        auth = None
        sandbox_config = common.config.Registry()
        self.client = common.rest.Client(sandbox_config.common.walle.api, auth=auth)
        self.projects = tuple(sandbox_config.common.walle.projects)

    def _get_walle_hosts(self):
        res = set()
        cursor = 0

        query = dict(
            fields=["name", "status"],
            limit=10000,
            project__in=self.projects
        )

        bar = None
        while cursor != -1:
            query["cursor"] = cursor

            response = self.client.hosts.read(**query)
            if not bar:
                bar = console.ProgressBar(
                    "Getting Wall-e hosts at {}".format(self.client.host),
                    response["total"]
                )
            hosts = response.get("result", [])
            if not hosts:
                break

            res.update(
                hc.WalleHost(host["name"], host["status"]) for host in hosts
            )
            bar.add(len(hosts))
            cursor = response.get("next_cursor", -1)
        bar.finish()

        return res

    @singleton_property
    def hosts(self):
        return self._get_walle_hosts()


class Sandbox:
    def __init__(self):
        token = oauth.get_token(client_id=consts.OAUTH_CLIENT["id"], client_secret=consts.OAUTH_CLIENT["secret"])
        self.clients = [
            common.rest.Client(url + "/api/v1.0", auth=token) for key, url in consts.SANDBOX_BASE_URLS.items()
        ]

    @staticmethod
    def _get_sandbox_hosts_client(client):
        query = {
            "limit": 10000,
            "fields": [
                "tags",
                "fqdn",
                "alive",
                "last_activity",
            ]
        }
        res = client.client.read(**query)
        hosts = set()
        bar = console.ProgressBar(
            "Getting sandbox hosts at {}".format(client.host),
            len(res["items"])
        )
        for data in res["items"]:
            hosts.add(
                hc.SandboxHost(data["fqdn"], data["tags"], data["alive"], data["last_activity"])
            )
            bar.add(1)
        bar.finish()
        return hosts

    def get_all_hosts(self):
        with futures.ThreadPoolExecutor(max_workers=64) as executor:
            jobs = [
                executor.submit(self._get_sandbox_hosts_client, client) for client in self.clients
            ]
            return set(itertools.chain.from_iterable(job.result() for job in jobs))

    @singleton_property
    def hosts(self):
        return self.get_all_hosts()


class Samogon:
    def __init__(self):
        token = oauth.get_token(client_id=consts.OAUTH_CLIENT["id"], client_secret=consts.OAUTH_CLIENT["secret"])
        self.clients = {
            url: common.rest.Client(url, auth=token) for url in consts.SAMOGON_URLS
        }

    @staticmethod
    def _get_samogon_hosts(url, client):
        meta = client.meta.read()
        version = meta["value"]["package"]

        res = set()
        data = client.hosts.versions.read()
        hosts = data["value"]
        bar = console.ProgressBar(
            "Getting samogon hosts at {}".format(client.host),
            len(hosts)
        )
        for fqdn in hosts:
            pkg = hosts[fqdn]["package"]
            too_old = False
            host_pkg = pkg
            if pkg is None:
                too_old = True
            elif isinstance(pkg, str):
                if pkg == "unknown" or pkg == "too old info":
                    too_old = True
            elif isinstance(pkg, list):
                host_pkg = pkg[0]
                too_old = host_pkg != version
            else:
                too_old = True
            res.add(
                hc.SamogonHost(fqdn, host_pkg, too_old=too_old, is_virtual=('2' in url or '3' in url))
            )
            bar.add(1)
        bar.finish()

        return res

    def _get_samogon_hosts_all_urls(self):
        with futures.ThreadPoolExecutor(max_workers=64) as executor:
            jobs = [
                executor.submit(self._get_samogon_hosts, url, self.clients[url]) for url in self.clients
            ]
            return set(itertools.chain.from_iterable(job.result() for job in jobs))

    @singleton_property
    def hosts(self):
        return self._get_samogon_hosts_all_urls()


class Checker:
    def __init__(self):
        self.conductor = Conductor()
        self.walle = Walle()
        self.sandbox = Sandbox()
        self.samogon = Samogon()

    @singleton_property
    def conductor_hosts(self):
        return self.conductor.hosts

    @singleton_property
    def walle_hosts(self):
        return self.walle.hosts

    @singleton_property
    def sandbox_hosts(self):
        return self.sandbox.hosts

    @singleton_property
    def samogon_hosts(self):
        return self.samogon.hosts

    def _mark_conductor_windows_hosts(self):
        for chost in self.conductor_hosts:
            if chost in self.conductor.win_hosts:
                chost.is_windows = True
            else:
                chost.is_windows = False

    def _merge_hosts(self):
        data = list(self.conductor_hosts) + list(self.walle_hosts) + list(self.sandbox_hosts) + list(self.samogon_hosts)
        data = sorted(data, key=lambda host: host.fqdn)

        res = []
        for key, group in itertools.groupby(data, lambda host: host.fqdn):
            merged = hc.MergedHost(list(group))
            res.append(merged)
        return res

    def get_problematic_hosts_table(self):
        table = []
        self._mark_conductor_windows_hosts()
        data = self._merge_hosts()
        for i, host in enumerate(data):
            if i == 0:
                table.append(list(
                    host.describe().keys()
                ))
            if host.is_problematic():
                if host.in_conductor:
                    host.conductor.group = self.conductor.resolve_group_id(host.conductor.group)
                table.append(list(
                    host.describe().values()
                ))
        return table

    @staticmethod
    def print_table(lines, title="", file=sys.stdout):
        file.write(title)
        file.write("\n")
        table = terminaltables.SingleTable(lines)
        file.write(table.table)
        file.write("\n")

    def check(self):
        lines = self.get_problematic_hosts_table()
        self.print_table(lines, "Samogon old version or absence in at least one system")
        print("Total records: {}".format(len(lines)))


def main(args):
    ch = Checker()
    start = time.time()

    ch.check()

    finish = time.time()
    print("Total running time: {:.3f}s".format(finish - start))


def setup_parser(parser):
    parser.set_defaults(func=main)
