# coding: utf-8
from __future__ import division, print_function, unicode_literals

import io
import sys
import math
import collections
import distutils.version

from six.moves.urllib import parse as urlparse

import paramiko
import paramiko.client
import concurrent.futures as futures

from sandbox.common import itertools as citertools

from .. import common_args, consts, utils


SSHCheckResult = collections.namedtuple("SSHCheckResult", "id error")
Host = collections.namedtuple("Host", "id fqdn")

NECESSARY_HOST_OPERATIONS = ["reload", "reboot"]


def naturally_sorted(hosts):
    return sorted(hosts, key=lambda t: distutils.version.LooseVersion(t.id))


def recheck_dead_hosts(client, hosts):
    dead = set()
    for chunk in citertools.chunker(hosts, consts.DEAD_CLIENTS_CHUNK_SIZE):
        data = client.client.read(id=",".join(chunk), limit=consts.DEAD_CLIENTS_CHUNK_SIZE)["items"]
        dead.update(
            Host(id=item["id"], fqdn=item["fqdn"])
            for item in data
            if not item["alive"]
        )

    return naturally_sorted(dead)


def fetch_dead_hosts(client, tag_query):
    data = client.client.read(limit=consts.DEAD_CLIENTS_LIMIT, alive=False, tags=str(tag_query or ""))

    return naturally_sorted(
        Host(id=item["id"], fqdn=item["fqdn"])
        for item in data["items"]
    )


def send_command_to_hosts(client, hosts, command):
    client.batch.clients[command].update(
        id=[_.id for _ in hosts],
        comment="{} sent from \"excavator\" tool".format(command)
    )


def ssh_checker(host, samogon_key):
    try:
        with paramiko.SSHClient() as client:
            client.set_missing_host_key_policy(paramiko.AutoAddPolicy())
            client.connect(host.fqdn, timeout=consts.SSH_TIMEOUT)
            client.exec_command(
                (
                    "sudo rm /opt/sandbox/run/client_check_stop; "
                    "sudo /samogon/{}/cli startall -f"
                ).format(samogon_key),
                timeout=consts.SSH_TIMEOUT,
            )

    except (paramiko.SSHException, paramiko.client.NoValidConnectionsError) as exc:
        return SSHCheckResult(id=host.id, error=exc)

    else:
        return SSHCheckResult(id=host.id, error=None)


def inline_progress(current, total, symbolic_results):
    print(
        "\r{}/{} {}".format(current, total, symbolic_results.getvalue()),
        end="\n" if current == total else "",
        file=sys.stderr
    )
    sys.stderr.flush()


def resurrect_hosts(hosts, samogon_key):
    symbolic_results = io.StringIO()
    failures = []
    pool_size = int(math.ceil(len(hosts) / 2))

    with futures.ThreadPoolExecutor(max_workers=pool_size) as pool:
        future_results = [
            pool.submit(ssh_checker, host, samogon_key)
            for host in hosts
        ]
        for i, future in enumerate(futures.as_completed(future_results)):
            result = future.result()
            if result.error is not None:
                failures.append(result)
            symbolic_results.write("+" if result.error is None else "-")
            inline_progress(i + 1, len(hosts), symbolic_results)

    if failures:
        print(
            "Failed to revive:\n  {}".format(
                "\n  ".join(
                    "{}: {}".format(item.id, item.error)
                    for item in naturally_sorted(failures)
                )
            )
        )


def main(args):
    clients_filter = common_args.ClientsFilter.create(args)
    api = common_args.API.create(args)
    client = utils.sandbox_client(api.base_url, auth=api.auth)

    hosts = args.hosts
    if hosts:
        hosts = recheck_dead_hosts(client, hosts)
    else:
        hosts = fetch_dead_hosts(client, clients_filter.tag_query)

    if not hosts:
        print("Nothing to revive -- all of clients provided are up and running")
        return

    print(
        "Hosts to operate on:\n  {}".format(
            "\n  ".join(_.id for _ in hosts)
        )
    )

    for command in NECESSARY_HOST_OPERATIONS:
        if getattr(args, command, False) or args.auto:
            send_command_to_hosts(client, hosts, command)

    installation = next(
        iter(
            cluster
            for cluster, url in consts.SANDBOX_BASE_URLS.items()
            if urlparse.urlparse(url).hostname == urlparse.urlparse(api.base_url).hostname
        ),
        None
    )
    samogon_key = consts.SANDBOX_DEPLOY_KEYS.get(installation)
    if samogon_key is None:
        print("Skipping resurrection: failed to determine Samogon cluster based on API URL")
    else:
        resurrect_hosts(hosts, samogon_key)


def setup_parser(parser):
    for command in NECESSARY_HOST_OPERATIONS:
        parser.add_argument(
            "--{}".format(command), action="store_true", help="Send {} command to hosts".format(command.upper())
        )

    parser.add_argument(
        "--auto", action="store_true", help=(
            "Schedule RELOAD/REBOOT commands and launch all servants on every host. "
            "In case no hosts are supplied, obtain them from API"
        )
    )
    parser.add_argument("--hosts", nargs="*", help="Host identifiers")
    common_args.ClientsFilter.setup(parser)
    common_args.API.setup(parser)

    parser.set_defaults(func=main)
