# coding: utf-8
import argparse
import datetime
from distutils import util

from sandbox.scripts.tools.excavator import consts, utils


def restart_task(args):
    base_url = consts.SANDBOX_BASE_URLS["production"]
    client = utils.sandbox_client(base_url)

    while True:
        answer = input("Continue? [Y/n] ").strip() or "yes"
        try:
            if util.strtobool(answer):
                break
            else:
                return
        except Exception as exc:
            print(exc)
            pass

    ids = args.ids
    limit = 100000
    params = {"limit": limit, "status": "EXCEPTION", "offset": 0, "fields": ["id", "status", "owner", "type", "url"]}
    if ids:
        params.update({"id": ids})
    else:
        date_range = args.start + ".." + \
            (args.end or datetime.datetime.utcnow().replace(tzinfo=datetime.timezone.utc).isoformat())
        params.update({"type": args.task_type, "owner": args.owner, "created": date_range})

    response = client.task.read(params=params)
    tasks = response["items"]
    if response["total"] > len(tasks):
        print(f"There are more than {limit} tasks to restart")

    print("Tasks to restart:")
    for task in tasks:
        print(f"{' '*7}Task: id={task['id']}, title={task['type']}, owner={task['owner']}")

    batch = 100
    for start in range(0, len(tasks), batch):
        resp = client.batch.tasks.start.update(id=[task['id'] for task in tasks[start:start+batch]])

        for task, result in zip(tasks[start:start+batch], resp):
            status = result["status"]
            print(f"{status}: {task['url']}")
            if status != "SUCCESS":
                print("    {}".format(result["message"]))


def setup_parser(parser):
    group = parser.add_argument_group("Filter Parameters")
    group.add_argument("-t", "--task_type", type=str, help="Task type")

    group.add_argument("-s", "--start", type=str, help="Task creation time since. Date string in ISO 8601 format")

    group.add_argument("-e", "--end", type=str, help="Task creation time till. Date string in ISO 8601 format")

    group = parser.add_mutually_exclusive_group(required=True)

    group.add_argument("-i", "--ids", type=int, default=[], nargs="*", help="Space separated list of task ids")

    group.add_argument("-o", "--owner", type=str, help="Task owner")

    parser.set_defaults(func=main)


def main(args):
    parser = argparse.ArgumentParser()
    varargs = vars(args)
    if not varargs["ids"] and (varargs["owner"] and (varargs["start"] is None or varargs["task_type"] is None)):
        raise parser.error("--start and --task_type is required with --owner")

    restart_task(args)
