# coding: utf-8
from __future__ import division, print_function, unicode_literals

import argparse

from sandbox.common.types import client as ctc

from . import consts


class ArgumentsGroup(object):
    __slots__ = ()

    @staticmethod
    def setup(args):
        pass

    @classmethod
    def create(cls, args):
        obj = cls()
        for field in cls.__slots__:
            setattr(obj, field, getattr(args, field))
        return obj


class API(ArgumentsGroup):
    __slots__ = ("base_url", "auth", "version")

    @staticmethod
    def setup(parser):
        api_group = parser.add_argument_group("Sandbox API")

        env_group = api_group.add_mutually_exclusive_group()
        env_group.add_argument("--url", required=False, help="Base Sandbox URL")
        env_group.add_argument(
            "--env", default="production", choices=tuple(consts.SANDBOX_BASE_URLS), help="Sandbox environment"
        )
        api_group.add_argument("--no-auth", dest="auth", action="store_false", help="Disable authentication")

    @classmethod
    def create(cls, args):
        obj = cls()
        if args.url:
            obj.base_url = args.url
        else:
            obj.base_url = consts.SANDBOX_BASE_URLS[args.env]
        obj.auth = args.auth
        return obj


class Task(ArgumentsGroup):
    __slots__ = ("owner", "draft", "arcadia_patch")

    @staticmethod
    def setup(parser):
        task_group = parser.add_argument_group("Task parameters")
        task_group.add_argument("-o", "--owner", default="SANDBOX", type=str, help="Task owner, default SANDBOX")
        task_group.add_argument("-d", "--draft", action="store_true", help="Do not run the tasks")
        task_group.add_argument("-p", "--arcadia_patch", help="Apply patch. Doc: https://nda.ya.ru/3QTTV4")


class ClientsFilter(ArgumentsGroup):
    __slots__ = ("tags", "all")

    _EXCLUDED_CLIENTS = ctc.Query("~MAINTENANCE & ~NEW")

    @staticmethod
    def _validate_tags(tags):
        tags = tags.strip()
        if tags:
            try:
                return ctc.Query(tags)
            except ValueError as exc:
                raise argparse.ArgumentTypeError(exc[0])

    @classmethod
    def setup(cls, parser):
        group = parser.add_argument_group("Clients filtering parameters")
        group.add_argument("-t", "--tags", default="", type=cls._validate_tags, help="Tag expression to filter")
        group.add_argument("-a", "--all", action="store_true", help="Do not skip clients with MAINTENANCE and NEW tags")

    @property
    def tag_query(self):
        tag_query = None
        if self.tags:
            tag_query = self.tags
            if not self.all:
                tag_query &= self._EXCLUDED_CLIENTS
        elif not self.all:
            tag_query = self._EXCLUDED_CLIENTS
        return tag_query
