class ABCHost:
    def __init__(self, fqdn):
        self.fqdn = fqdn

    def __hash__(self):
        return str.__hash__(self.fqdn)

    def __eq__(self, other):
        return self.fqdn == other.fqdn

    def __str__(self):
        return self.fqdn

    def __repr__(self):
        return self.fqdn


class ConductorHost(ABCHost):
    def __init__(self, fqdn, inv, group):
        super(ConductorHost, self).__init__(fqdn)
        self.inv = inv
        self.group = group
        self.is_windows = None


class SandboxHost(ABCHost):
    def __init__(self, fqdn, tags, alive, last_activity):
        super(SandboxHost, self).__init__(fqdn)
        self.tags = tags
        self.alive = alive
        self.last_activity = last_activity


class WalleHost(ABCHost):
    def __init__(self, fqdn, status):
        super(WalleHost, self).__init__(fqdn)
        self.status = status


class SamogonHost(ABCHost):
    def __init__(self, fqdn, version, too_old=False, is_virtual=False):
        super(SamogonHost, self).__init__(fqdn)
        self.version = version
        self.too_old = too_old
        self.is_virtual = is_virtual


class MergedHost:
    def __init__(self, hosts):

        fqdns = {host.fqdn for host in hosts if host}
        if not hosts or len(fqdns) != 1:
            raise ValueError("fqdns should match in order for hosts to be merged")

        self.fqdn = hosts[0].fqdn
        self.in_conductor = self.in_sandbox = self.in_walle = self.in_samogon = False
        for host in hosts:
            if isinstance(host, ConductorHost):
                self.in_conductor = True
                self.conductor = host
            if isinstance(host, SandboxHost):
                self.in_sandbox = True
                self.sandbox = host
            if isinstance(host, WalleHost):
                self.in_walle = True
                self.walle = host
            if isinstance(host, SamogonHost):
                self.in_samogon = True
                self.samogon = host

    @staticmethod
    def _ok(arg=True):
        return "✓" if arg else ""

    def _is_present_everywhere(self):
        all_but_walle = self.in_samogon and self.in_sandbox and self.in_conductor
        if self.in_conductor and self.conductor.is_windows and all_but_walle:
            return True
        if self.in_samogon and self.samogon.is_virtual and self.in_sandbox and self.sandbox.alive:
            return True
        return all_but_walle and self.in_walle

    def _is_samogon_version_uptd(self):
        if not self.in_samogon:
            return False
        return not self.samogon.too_old

    def is_problematic(self):
        checks = self._is_samogon_version_uptd() and self._is_present_everywhere()
        return not checks

    def describe(self):
        info = dict(
            fqdn=self.fqdn,
            group="",
            in_conductor=self._ok(False),
            in_walle=self._ok(False),
            in_sandbox=self._ok(False),
            in_samogon=self._ok(False),
            sandbox_alive=self._ok(False),
            last_activity="",
            samogon_version="",
            too_old=self._ok(False)
        )

        if self.in_conductor:
            info["group"] = self.conductor.group
            info["in_conductor"] = self._ok()
        if self.in_walle:
            info["in_walle"] = self._ok()
        if self.in_sandbox:
            info["in_sandbox"] = self._ok()
            info["sandbox_alive"] = self._ok(self.sandbox.alive)
            info["last_activity"] = self.sandbox.last_activity
        if self.in_samogon:
            info["in_samogon"] = self._ok()
            info["too_old"] = self._ok(self.samogon.too_old)
            if self.samogon.too_old:
                info["samogon_version"] = self.samogon.version

        return info
