# coding: utf-8
from __future__ import division, print_function, unicode_literals

import re
import argparse
import collections

from six.moves import urllib
from library.python import oauth

from .consts import OAUTH_CLIENT

UNITS = ((7 * 24 * 3600, "w"), (24 * 3600, "d"), (3600, "h"), (60, "m"), (1, "s"))
ITEM_RE = re.compile(r"(?:([a-z0-9]+)(?:[\s,]+)?)+?", re.IGNORECASE)


def list_parse(raw, allow_empty=False, allowed=None, unique=True):
    parsed = ITEM_RE.findall(raw)
    if not (parsed or allow_empty):
        raise argparse.ArgumentTypeError("empty list is not allowed")

    if allowed:
        illegal = set(parsed) - set(allowed)
        if illegal:
            raise argparse.ArgumentTypeError("illegal values: %s" % sorted(illegal))

    if unique:
        parsed = list(collections.OrderedDict.fromkeys(parsed))

    return parsed


def duration_string(td):

    remainder = td.total_seconds()
    result = []

    for lenght, unit in UNITS:
        num, remainder = divmod(remainder, lenght)
        if num:
            result.append("%d%s" % (num, unit))

    return " ".join(result)


def oauth_token():
    return oauth.get_token(client_id=OAUTH_CLIENT["id"], client_secret=OAUTH_CLIENT["secret"])


def sandbox_client(base_url, version="1.0", auth=True):
    from sandbox.common import rest

    parsed_base_url = urllib.parse.urlparse(base_url)

    token = None
    if auth:
        token = oauth_token()

    url = parsed_base_url._replace(path="/api/v%s" % version)
    url = urllib.parse.urlunparse(url)

    return rest.Client(url, auth=token)
