"""
Upload file(s) to Sandbox
"""

import os
import re
import sys
import logging
import argparse
import urllib.parse
import operator as op
import itertools as it
import contextlib

import sandbox.common.log as common_log
import sandbox.common.auth as common_auth
import sandbox.common.rest as common_rest
import sandbox.common.format as common_format
import sandbox.common.upload as common_upload
import sandbox.common.console as common_console
import sandbox.common.types.misc as ctm

# noinspection PyUnresolvedReferences
from library.python import oauth
from library.python import svn_version

from sandbox.scripts.tools.upload import uploader

MAX_OPEN_FILES = 1000  #: Amount of files can be opened simultaneously

# Scopes:
#   sandbox:api
CLIENT_ID = "cb8c7d39156c45b58538db30c9a642ba"
CLIENT_SECRET = "91c46a472de94b329bdfed01c18c82f4"


def handle_files(args: [str]):
    ret = {}
    cwd = os.curdir
    unrel_re = re.compile(r"^(\.+/?)+")
    for root, _, files in it.chain.from_iterable(map(
        lambda f: os.walk(f) if os.path.isdir(f) else ((os.path.dirname(f), tuple(), (os.path.basename(f),)),),
        args
    )):
        for subf in files:
            fname = os.path.join(root, subf)
            rpath = os.path.realpath(fname)
            if rpath not in ret:
                fh = open(fname, "rb")
                fh.seek(0, os.SEEK_END)
                ret[rpath] = common_upload.HTTPHandle.FileMeta(
                    fh if len(ret) < MAX_OPEN_FILES else rpath, fh.tell(), unrel_re.sub("", os.path.relpath(fname, cwd))
                )
                fh.seek(0)

    return sorted(ret.values(), key=op.itemgetter(2))


def handle_args() -> argparse.Namespace:
    parsed_url = list(urllib.parse.urlparse(common_rest.Client.DEFAULT_BASE_URL))
    parsed_url[0] = "https"    # scheme
    parsed_url[2] = ""         # path
    default_url = urllib.parse.urlunparse(parsed_url)

    parsed_url[0] = "http"     # scheme
    parsed_url[1] = ".".join(["proxy", parsed_url[1]])
    default_proxy_url = urllib.parse.urlunparse(parsed_url)

    parser = argparse.ArgumentParser(
        formatter_class=lambda *args, **kwargs: argparse.ArgumentDefaultsHelpFormatter(*args, width=120, **kwargs),
        description="Sandbox data upload tool."
    )
    parser.add_argument(
        "-v", "--verbosity",
        action="count", help="Increase output verbosity"
    )
    parser.add_argument(
        "-q", "--quiet",
        action="store_true", help="Silent or quiet mode. Don't show progress meter and runtime stages"
    )
    parser.add_argument(
        "-I", "--noninteractive",
        default=not sys.stdout.isatty(),
        action="store_true", help="Do try to interactively ask authorization information"
    )
    parser.add_argument(
        "-D", "--dump",
        default="", type=str,
        help="Dump created resource metadata in JSON format to the file specified (or STDOUT for '-')"
    )
    parser.add_argument(
        "-t", "--type",
        default="OTHER_RESOURCE", type=str, help="Resource type to register"
    )
    parser.add_argument(
        "-d", "--description",
        default="", type=str, help="Resource description to set"
    )
    parser.add_argument(
        "-A", "--attribute",
        type=str, nargs='*', help="Additional resource attribute(s) to set in form `key=value`"
    )
    parser.add_argument(
        "-a", "--arch",
        default=ctm.OSFamily.ANY, type=str, choices=list(ctm.OSFamily), help="Resource architecture to register"
    )
    parser.add_argument(
        "-U", "--url",
        default=default_url, type=str, help="Sandbox main URL to communicate with"
    )
    parser.add_argument(
        "-P", "--proxy-url",
        default=default_proxy_url, type=str, help="Sandbox proxy URL to upload the actual data with"
    )
    parser.add_argument(
        "-w", "--call-wait",
        default=None, type=int, help="Total response wait time limit per RPC call in seconds. No limit by default"
    )
    parser.add_argument(
        "-o", "--owner",
        required=True, default=None, type=str, help="Task/resource owner to be set. Uses user name if not specified"
    )
    parser.add_argument(
        "-T", "--oauth-token",
        default=None, type=str,
        help="OAuth token to authenticate instead of SSH key or interactive mode. "
        "In case of '-' specified, the token will be taken from 'OAUTH_TOKEN' environment variable"
    )
    parser.add_argument(
        "-s", "--stream-name",
        default="stream.txt", type=str,
        help="File name to be created in case of content provided via STDIN stream"
    )
    parser.add_argument(
        "-l", "--stream-limit",
        default="1G", type=str,
        help="Stream limit in case of reading content from STDIN instead of file(s) from positional arguments. "
        "Possible suffixes are K, M, and G (powers of 1024)"
    )
    parser.add_argument(
        "-M", "--mds",
        default=None, action="store_true",
        help="Upload to MDS"
    )
    parser.add_argument(
        "-N", "--no-auth",
        default=None, action="store_true",
        help="No authorization"
    )
    parser.add_argument(
        "-L", "--log-dir",
        default=os.path.expanduser("~"), type=str, help="Directory for log"
    )
    parser.add_argument(
        "files", metavar="FILE", nargs="+",
        help="File(s)/directory(ies) list to upload. The script will read STDIN in case '-' file specified"
    )
    args = parser.parse_args()
    args.url = (args.url or default_url).rstrip("/")
    args.oauth_token = os.environ[ctm.OAUTH_TOKEN_ENV_NAME] if args.oauth_token == "-" else args.oauth_token
    args.stream_limit = common_format.str2size(args.stream_limit)
    if "-" in args.files:
        if len(args.files) != 1:
            raise ValueError("It is only one STDIN stream allowed in positional arguments")
        args.files = None
    return args


def main():
    args = handle_args()
    cz = common_console.AnsiColorizer()
    # File handle to output verbose information like progress meter and so on.
    pgfh = open(os.devnull, "w") if args.quiet or (args.verbosity or 0) > 0 else sys.stderr
    logfile = common_log.script_log(args.verbosity, args.log_dir, "upload.log")
    version = svn_version.svn_revision()

    logging.getLogger(__name__).info("Sandbox upload binary r%s started.", version)
    print(cz.black(f"Sandbox upload binary r{version}, log at '{logfile}'"), file=pgfh)

    handle_class = common_upload.MDSHandle if args.mds else common_upload.HTTPHandle
    token = common_console.Token(args.url, None, not args.noninteractive, pgfh)
    if args.no_auth:
        auth = common_auth.NoAuth()
    elif args.oauth_token and token.check(args.oauth_token):
        auth = args.oauth_token
    else:
        auth = oauth.get_token(CLIENT_ID, CLIENT_SECRET)
    resource_meta = common_upload.HTTPHandle.ResourceMeta(
        args.type, args.arch, args.owner, args.description,
        {
            k.strip(): v.strip()
            for k, v in (a.split("=") for a in common_format.chain(*(_.split(",") for _ in args.attribute)))
        } if args.attribute else None,
        release_to_yd=False
    )
    if args.files:
        payload_meta = handle_files(args.files)
    else:
        payload_meta = [common_upload.HTTPHandle.StreamMeta(sys.stdin, args.stream_limit, args.stream_name)]
    sandbox_uploader = uploader.SandboxUploader(
        handle_class, resource_meta, auth, args.url, args.proxy_url, payload_meta, args.call_wait
    )
    if args.dump:
        with (open(args.dump, "w") if args.dump != "-" else contextlib.closing(sys.stdout)) as dump_target:
            sandbox_uploader.upload(pgfh, dump_target)
    else:
        resource_id = sandbox_uploader.upload(pgfh)
        if args.quiet:
            print(resource_id)
    return 0


if __name__ == "__main__":
    try:
        sys.exit(main())
    except KeyboardInterrupt:
        print("Program interrupted by user.", file=sys.stderr)
        sys.exit(2)
