import copy
import itertools
import json
from typing import Type, Union, Optional, IO, Collection
# noinspection PyUnresolvedReferences
import progressbar as pb

import sandbox.common.auth as common_auth
import sandbox.common.format as common_format
import sandbox.common.upload as common_upload
import sandbox.common.console as common_console
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
import sandbox.common.types.resource as ctr


class SandboxUploader:
    @staticmethod
    def _get_task_palette() -> dict[str, str]:
        cz = common_console.AnsiColorizer()
        task_palette = {
            k: cz.colorize(k, c)
            for c, k in itertools.chain.from_iterable((
                zip(itertools.repeat("green"), ctt.Status.Group.FINISH),
                zip(itertools.repeat("blue"), ctt.Status.Group.EXECUTE),
                zip(itertools.repeat("yellow"), ctt.Status.Group.BREAK),
                zip(itertools.repeat("white"), ctt.Status.Group.QUEUE),
                zip(itertools.repeat("black"), ctt.Status.Group.DRAFT),
            ))
        }
        task_palette.update({
            k: cz.colorize(c, k) for k, c in zip(("black", "red"), (ctt.Status.DELETED, ctt.Status.FAILURE))
        })
        return task_palette

    @staticmethod
    def _get_resource_palette() -> dict[str, str]:
        cz = common_console.AnsiColorizer()
        return {
            k: cz.colorize(k, c)
            for c, k in itertools.chain.from_iterable((
                zip(
                    ("white", "green", "red", "black"),
                    (ctr.State.NOT_READY, ctr.State.READY, ctr.State.BROKEN, ctr.State.DELETED)
                ),
            ))
        }

    def __init__(
        self, handle_class: Type[common_upload.HTTPHandle],
        resource_meta: common_upload.HTTPHandle.ResourceMeta,
        auth: Union[str, common_auth.Authentication], sandbox_url: str, sandbox_proxy_url: str,
        payload_meta: Union[common_upload.HTTPHandle.StreamMeta, Collection[common_upload.HTTPHandle.FileMeta]],
        wait: Optional[int] = None
    ):
        self.task_palette = self._get_task_palette()
        self.resource_palette = self._get_resource_palette()
        self.sandbox_url = sandbox_url
        self.sandbox_proxy_url = sandbox_proxy_url
        self._is_stream = isinstance(payload_meta, common_upload.HTTPHandle.StreamMeta)
        self.upload_handler = handle_class(
            resource_meta, auth, sandbox_url, sandbox_proxy_url, wait,
            *payload_meta if not self._is_stream else payload_meta
        )

    def upload(self, output: IO, dump: Optional[IO] = None) -> int:
        cz = common_console.AnsiColorizer()
        state, last_state, last_state_copy = (None,) * 3
        cop = common_console.Operation(output)
        for state in self.upload_handler():
            if last_state != state:  # Last operation stopped.
                if isinstance(last_state, ctm.Upload.Check):
                    cop.long.intermediate(
                        f"{last_state.amount} file(s) found totally for {common_format.size2str(last_state.size)}."
                    )
            else:  # Operation continues.
                if isinstance(state, ctm.Upload.Prepare):
                    if last_state_copy.task_id != state.task_id:
                        link = f"{self.sandbox_url}/task/{state.task_id}"
                        cop.long.intermediate(
                            f"Task #{cz.white(state.task_id)} created: {cz.blue(link)}"
                        )
                    if last_state_copy.resource_id != state.resource_id:
                        link = f"{self.sandbox_url}/resource/{state.resource_id}"
                        cop.long.intermediate(
                            f"Resource #{cz.white(state.resource_id)} registered: {cz.blue(link)}"
                        )
                if isinstance(state, ctm.Upload.DataTransfer):
                    cop.pbar.update(state.done)
                if isinstance(state, ctm.Upload.ShareResource):
                    cop.long.intermediate(
                        f"Resource currently is in '{self.resource_palette[state.resource_state]}' state.", cr=True
                    )
                elif isinstance(state, ctm.Upload.Share):
                    cop.long.intermediate(
                        f"Task currently is in '{self.task_palette[state.task_state]}' state.    ", cr=True
                    )

            if last_state != state:  # Start new operation report.
                cop.finish(last_state.done if isinstance(last_state, ctm.Upload.DataTransfer) else None)

                if isinstance(state, ctm.Upload.Check):
                    cop.long = "Calculating total files size"

                if isinstance(state, ctm.Upload.Prepare):
                    cop.long = "Preparing upload task"

                if isinstance(state, ctm.Upload.DataTransfer):  # Prepare started
                    size_label = pb.FormatLabel("%(cursize)s/%(maxsize)s")
                    size_label.mapping["cursize"] = ("value", lambda x: common_format.size2str(x))
                    size_label.mapping["maxsize"] = ("max_value", lambda x: common_format.size2str(x))
                    if self._is_stream:
                        cop.pbar = (
                            "", state.total,
                            [
                                "[", pb.AnimatedMarker(), "] ",
                                "Uploading ",
                                size_label, " | ",
                                pb.Timer(), " | ",
                                pb.FileTransferSpeed()
                            ],
                        )
                    else:
                        cop.pbar = (
                            "", state.total,
                            [
                                "Uploading ",
                                pb.Bar(), " ",
                                pb.Percentage(), " | ",
                                size_label, " | ",
                                pb.Timer(), " | ",
                                pb.ETA(), " |",
                                pb.FileTransferSpeed()
                            ],
                        )

                if isinstance(state, ctm.Upload.Share):
                    cop.long = "Sharing uploaded data"

            last_state = state
            last_state_copy = copy.deepcopy(state)

        if isinstance(state, ctm.Upload.Share):
            if isinstance(state, ctm.Upload.ShareResource):
                cop.long.intermediate(
                    f"Resource currently is in '{self.resource_palette[state.resource_state]}' state.    "
                )
                cop.long.intermediate(f"Resource download link: {cz.blue(state.meta['mds']['url'])}")
            else:
                cop.long.intermediate(
                    f"Task currently is in '{self.task_palette[state.task_state]}' state.    "
                )
                cop.long.intermediate(
                    "Resource download link: {}".format(
                        cz.blue("/".join([self.sandbox_proxy_url.rstrip("/"), str(state.meta["id"])]))
                    )
                )
            msg = None
            if state.skynet_id:
                msg = f"Skybone ID is {cz.white(state.skynet_id)}"
            if state.md5sum:
                if msg:
                    msg += ", "
                msg += f"MD5 checksum is {cz.white(state.md5sum)}"
            if msg:
                cop.long.intermediate(msg)
        cop.finish()
        if dump:
            json.dump(state.meta, dump, indent=4, sort_keys=True, separators=(",", ": "))
            print("", file=dump)
        return state.meta["id"]
