"""
Build and freeze virtualenv package for Sandbox for this host's platform

Build steps include:
- Install pre-requisites such as installers themselves and
    requirements from some packages' setup.py;
- Install all packages but these banned for this platform;
- If asked, install packages from wheels stored in Sandbox;
- Perform different hacky actions, including, but not limited to,
    installing platform-specific packages and patching existing ones;
- Save this binary's svn revision as virtualenv's revision;
- Make virtualenv relocatable and write it to a compressed archive.
"""

import os
import sys
import argparse
import subprocess as sp

import hacks
import utils

import pathlib2
from library.python import svn_version

from sandbox import common
import sandbox.common.types.client as ctc
from sandbox.sandboxsdk import environments

# Certain packages need to be installed first,
# because others depend on their existence (may be seen in their setup.py).
# Their versions are fetched from requirements.txt

# FIXME: Some of latest MacOS hosts don't have SSL properly configured, which leads to errors during
# FIXME: dependencies fetching/locating stage: https://bugs.python.org/issue28150 (applies for Python 2 as well).
# FIXME: If that is still the case, add the dependency to the second step
INSTALLATION_STEPS = [
    (
        "pip",
        "setuptools",
    ),
    (
        "numpy",  # required by pandas: https://stackoverflow.com/q/57734032/1823304
        "six",  # required by protobuf
        "google-apputils",  # required by protobuf (see the FIXME rant above)
    ),
]

ARCHIVE_FILENAME_TEMPLATE = "sandbox-venv.{revision}.tar.gz"


class CustomFormatter(argparse.ArgumentDefaultsHelpFormatter):
    def __init__(self, *args, **kwargs):
        kwargs["width"] = 120
        super(CustomFormatter, self).__init__(*args, **kwargs)

    def _fill_text(self, text, width, indent):
        return "".join([indent + line for line in text.splitlines(True)])


class Platform(common.utils.Enum):
    DARWIN = None
    LINUX = None
    WINDOWS = None


class DirType(object):
    def __call__(self, path):
        res = force_abspath(path)
        res.mkdir(parents=True, exist_ok=True)
        return res


def force_abspath(path):
    return pathlib2.Path(path) if os.path.isabs(path) else pathlib2.Path.cwd() / path


def svn_revision():
    rev = max(svn_version.svn_revision(), 0)
    return str(rev)


def perform_different_hacks(venv, env, args, requirements):
    os_name, os_alias = utils.get_os_version()
    with common.console.LongOperation("Performing various platform-related hacks") as op:
        if os_name == Platform.LINUX:
            if os_alias in (ctc.Tag.LINUX_PRECISE, ctc.Tag.LINUX_TRUSTY):
                # This is done for SANDBOX-2337; likely a dependency of butterfly package,
                # which helps with web shell

                six_package = next(iter(filter(lambda s: s.split("==")[0] == "six", requirements)), None)
                if six_package is None:
                    raise RuntimeError("'six' package is missing in parsed requirements")
                packages = ("libsass==0.8.0", six_package)
                utils.install_requirements(venv, env, packages, args.log_directory, use_cflags_ldflags=True)

            if os_alias == ctc.Tag.LINUX_XENIAL:
                # Done in r3830360 for SANDBOX-5452!
                # This package seems to be only buildable (available, even?) for Xenial, so...
                utils.install_wheels(venv, env, ("pydevd==1.3.2-1",), args.log_directory, args.workdir, args.token)

                # TODO SANDBOX-4810: remove this after proxy servant is deployed as binary
                hacks.reinstall_uwsgi(venv, env, requirements, args.log_directory)

        if os_name == Platform.DARWIN:
            # This is done solely for Browser team and stems from SANDBOX-4921 and related tickets
            utils.install_requirements(venv, env, ("watchdog==0.8.3",), args.log_directory, use_cflags_ldflags=True)

        if os_name != Platform.DARWIN:
            # r1551684 is where it's started -- first for OS X only, then for all platforms but OS X
            # (weird, huh?). Chances are it doesn't do anything as of now
            op.intermediate("- Wipe installed bson module (may affect pymongo's performance)")
            hacks.wipe_bson_folder(venv, args.log_directory)

        # https://wiki.yandex-team.ru/security/ssl/sslclientfix/#vpython
        op.intermediate("- Patch certifi (install internal Yandex certificate)")
        hacks.patch_certifi(venv)


def do_build(args):
    env = utils.prepare_environment()
    os_name, os_alias = utils.get_os_version()

    requirements = utils.read_requirements(args.requirements)

    base_exclusions = [
        common.fs.read_file("exclusions/{}.txt".format(base_filename)).splitlines()
        for base_filename in ("all", os_name.lower())
    ]
    exclusions = set(utils.read_requirements(args.exclusions + base_exclusions))

    # TODO SANDBOX-4810: remove this after proxy servant is deployed as binary
    if os_alias != ctc.Tag.LINUX_XENIAL:
        for item in requirements:
            if "uwsgi" in item.lower():
                exclusions.add(item.split("==")[0])

    requirements = sorted(filter(
        lambda package: package.split("==")[0] not in exclusions,
        requirements
    ))

    requirements_path = os.path.join(str(args.workdir), "requirements-final.txt")
    with open(requirements_path, "w") as fd:
        fd.write("\n".join(requirements))

    if os_name == Platform.LINUX:
        packages = [_.strip() for _ in common.fs.read_file("packages/linux.txt").splitlines()]
        utils.check_debpackages_installed(packages)

    # Starting from OS X Mojave, linker may miss system libraries to link the ones being built against
    # (libgcc_s.10.4.dylib for one). In this case, either of two solutions works:
    #   * symbolic links to newer SDK runtime: ln -s /usr/lib/libSystem.B.dylib /usr/local/lib/libgcc_s.10.4.dylib
    #   * explicit usage of newer SDK (see below)
    if os_name == Platform.DARWIN:
        os.environ["MACOSX_DEPLOYMENT_TARGET"] = "10.6"

    with environments.VirtualEnvironment(working_dir=str(args.workdir), use_system=True, do_not_remove=True) as venv:
        for step in INSTALLATION_STEPS:
            packages = sorted(filter(
                lambda package: package.split("==")[0] in step,
                requirements
            ))
            utils.install_requirements(venv, env, packages, args.log_directory)

        utils.install_requirements(venv, env, requirements, args.log_directory)
        if args.path:
            utils.install_requirements(
                venv, env, args.path, args.log_directory,
                use_cflags_ldflags=True, pip_flags=("--no-dependencies",)
            )

        if args.wheel:
            utils.install_wheels(venv, env, args.wheel, args.log_directory, args.workdir, args.token)

        perform_different_hacks(venv, env, args, requirements)

        with open(os.path.join(venv.root_dir, ".revision"), "w") as f:
            f.write(args.revision)

        with common.console.LongOperation("Packing virtualenv"):
            target = force_abspath(args.output.format(revision=args.revision))

            # compileall output clogs stderr, so we have to redirect it somewhere
            # XXX: this will redirect any further stdout/stderr, including occasional tracebacks -- stay alert!
            logfile = args.log_directory / "venv-pack-virtualenv.out.log"
            logfile.touch()

            utils.redirect_std_file(sys.stdout, str(logfile))
            utils.redirect_std_file(sys.stderr, str(logfile))

            with open(os.path.join(venv.root_dir, "requirements.txt"), "w") as fileobj:
                sp.Popen([venv.executable, "-m", "pip", "freeze"], stdout=fileobj).wait()

            venv.pack_bundle(str(target), log_prefix=str(args.log_directory / "venv-pack-bundle"))

        print("Virtualenv archive saved to {}".format(target))


def parse_args():
    parser = argparse.ArgumentParser(
        description=__doc__, formatter_class=CustomFormatter,
        usage="{} -r requirements.txt --output .".format(sys.argv[0])
    )

    default_workdir = "./venv-workdir"
    parser.add_argument(
        "--workdir", help="Working directory for logs and all the trash",
        type=DirType(), default=default_workdir
    )

    default_logdir = os.path.join(default_workdir, "logs")
    parser.add_argument(
        "-l", "--log-directory", help="Custom directory for logs",
        type=DirType(), default=default_logdir
    )

    parser.add_argument(
        "-r", "--requirements", help="Path(s) to requirements.txt files",
        nargs="+", type=argparse.FileType("r"), required=True
    )
    parser.add_argument(
        "-w", "--wheel", help="Install additional wheels (name==version) after requirements are installed",
        nargs="*", default=[]
    )
    parser.add_argument(
        "-p", "--path", help="Install package from specified path(s) directly",
        nargs="*", default=[]
    )
    parser.add_argument(
        "-x", "--exclusions", help="Path(s) to newline-separated list of exclusions",
        nargs="*", default=[], type=argparse.FileType("r")
    )

    parser.add_argument(
        "-t", "--token",
        help="Sandbox API token for wheels downloading (optional); if omitted, will be retrieved from SSH key",
        default=None
    )

    parser.add_argument(
        "--revision", help="Virtual environment's version; binary's SVN revision usually goes here",
        default=svn_revision(),
    )

    parser.add_argument(
        "-o", "--output", help="File path for the resulting package", default=ARCHIVE_FILENAME_TEMPLATE,
    )

    return parser.parse_args()


def main():
    do_build(parse_args())


if __name__ == "__main__":
    main()
