import os
import getpass
import platform
import subprocess as sp
import collections

from sandbox import common
import sandbox.common.types.resource as ctr

OS = collections.namedtuple("OS", "platform alias")


def get_os_version():
    alias = common.platform.get_platform_alias(platform.platform())
    return OS(
        platform.system().upper(),
        common.platform.PLATFORM_TO_TAG.get(alias, alias)
    )


def prepare_environment():
    env = os.environ.copy()
    for env_key in ("LD_LIBRARY_PATH", "LIBRARY_PATH"):
        if env_key in env:
            env[env_key] = ":".join(filter(
                lambda x: not x.startswith("/skynet/"),
                env[env_key].split(":"))
            )
    return env


def check_debpackages_installed(required):
    packages = sp.check_output(["dpkg", "--get-selections"]).splitlines()
    installed = set(
        package.split(":")[0]  # libtevent0:amd64 -> libtevent0
        for package, status in (
            line.split()
            for line in packages
        )
        if status == "install"
    )
    difference = set(required) - installed
    if difference:
        raise RuntimeError(
            "Some Debian packages are missing, please execute:\n\tsudo apt update && sudo apt install {}".format(
                " ".join(sorted(difference))
            )
        )


def read_requirements(files):
    def transform(line):
        return line.split("#")[0].strip()

    requirements = set()
    for file_ in files:
        requirements.update(
            filter(None, map(transform, file_))
        )
    return list(requirements)


def redirect_std_file(std_file, path):
    old_fd = std_file.fileno()
    new_fd = os.open(path, os.O_WRONLY)
    os.lseek(new_fd, 0, os.SEEK_END)
    os.dup2(new_fd, old_fd)
    os.close(new_fd)


def install_wheels(venv, env, wheels, log_directory, workdir, token=None):
    checker = common.console.Token(
        "https://sandbox.yandex-team.ru", getpass.getuser(), interactive=False, fh=open(os.devnull, "w")
    )
    if token is None:
        token = checker.get_token_from_ssh(None)
    elif not checker.check(token):
        raise RuntimeError("Sandbox OAuth token is invalid!")

    wheels_dir = workdir / "virtualenv-builder-wheels"
    wheels_dir.mkdir(parents=True, exist_ok=True)
    for i, pypackage in enumerate(wheels):
        with common.console.LongOperation(
            "({}/{}) Installing {} from wheel".format(i + 1, len(wheels), pypackage)
        ) as op:
            op.intermediate("Retrieving Sandbox resource")
            rid, filename, md5sum = find_wheel(pypackage, token)
            if rid is None:
                raise RuntimeError("Didn't find any wheel for {}".format(pypackage))

            op.intermediate("Installing from resource #{}".format(rid))
            wheel_path = str(wheels_dir / filename)
            with open(wheel_path, "wb") as fd:
                for chunk in common.utils.checker_fetcher(
                    "https://proxy.sandbox.yandex-team.ru/{}".format(rid), md5=md5sum
                ):
                    fd.write(chunk)

            venv.pip(
                " ".join((
                    "--no-index",
                    "--find-links",
                    str(wheels_dir),
                    pypackage,
                )),
                extra_env=env.copy(),
                log_prefix=str(log_directory / "wheel-{}".format(pypackage))
            )


def find_wheel(package, token):
    name, version = package.split("==")
    api = common.rest.Client(auth=token)
    res = next(
        iter(
            api.resource.read(
                type="PYTHON_WHEEL",
                attrs=dict(name=name, version=version),
                state=ctr.State.READY,
                limit=1,
            ).get("items", [])
        ),
        None
    )
    if not res:
        return (None,) * 3

    return res["id"], res["file_name"], res["md5"]


def install_requirements(venv, env, requirements, log_directory, use_cflags_ldflags=False, pip_flags=()):
    command = " ".join(common.utils.chain(requirements, pip_flags))
    with common.console.LongOperation(
        "Installing {}".format(command)
    ):
        venv.pip(
            command, extra_env=env.copy(), use_cflags_ldflags=use_cflags_ldflags,
            log_prefix=str(log_directory / "install-packages")  # in case it's a filesystem path
        )
