#!/usr/bin/env python
from __future__ import print_function, unicode_literals

import io
import os
import re
import sys
import shutil
import string
import getpass
import argparse
import tempfile


def print_and_exit(message, exit_code=1):
    print(message, file=sys.stderr)
    sys.exit(exit_code)


def get_arcadia_root():
    path = os.path.dirname(os.path.abspath(__file__))
    for _ in range(4):
        # noinspection PyBroadException
        try:
            if os.path.exists(os.path.join(path, ".arcadia.root")):
                return path
        except Exception:
            pass
        path = os.path.dirname(path)
    print_and_exit("Could not find Arcadia root")


def add_to_macro(yamake_path, macro, item):

    with open(yamake_path, "rb") as f:
        yamake_content = f.read().decode("utf-8")

    single_line = yamake_content.replace('\n', '|')

    regex = r"{}\((.+?)\)".format(macro)
    matches = list(re.finditer(regex, single_line))

    if len(matches) > 1:
        print_and_exit("Expected a single {} macro in {}".format(macro, yamake_path))

    match = matches[0]

    items = [path for path in (x.strip() for x in match.group(1).split('|')) if path]
    items.append(item)
    items.sort()

    output = io.StringIO()
    output.write(yamake_content[:match.start()])
    output.write("{}(\n    {}\n)".format(macro, "\n    ".join(items)))
    output.write(yamake_content[match.end():])

    with open(yamake_path, "w") as f:
        # noinspection PyTypeChecker
        f.write(output.getvalue())


def main():
    parser = argparse.ArgumentParser(prog="bootstrap")
    parser.add_argument(
        "--name", help="override the real login with an arbitrary string", required=False, default=None
    )
    args = parser.parse_args()

    if os.getuid() == 0:
        print_and_exit("Please do not run this script as root!")

    # Initialise

    root_dir = get_arcadia_root()
    task_dir = os.path.join(root_dir, "sandbox/projects/tutorial")
    if not os.path.exists(task_dir):
        print_and_exit("`{}` does not exist".format(task_dir))
    template_dir = os.path.join(os.path.dirname(os.path.abspath(__file__)), "templates")

    login = args.name or getpass.getuser()
    safe_login = re.sub(r'[^a-z0-9]', '', login, flags=re.I).lower()
    if not safe_login:
        print_and_exit("Login must contain at least one alphanumeric character")

    package_name = "hello_{}".format(safe_login)

    destination = os.path.join(task_dir, package_name)
    if os.path.exists(destination):
        print_and_exit(
            "`{}` already exists.\nUse --name to pick an alternative task name".format(os.path.relpath(destination))
        )

    class_name = "Hello{}".format(safe_login.capitalize())

    # Prepare sources from templates

    tmp_dir = tempfile.mkdtemp()

    package_dir = os.path.join(tmp_dir, package_name)
    os.mkdir(package_dir)

    with open(os.path.join(template_dir, "__init__.tpl")) as tpl:
        src = string.Template(tpl.read()).substitute(login=login, class_name=class_name)
        with open(os.path.join(package_dir, "__init__.py"), "w") as f:
            f.write(src)

    with open(os.path.join(template_dir, "ya.make.tpl")) as tpl:
        src = string.Template(tpl.read()).substitute(package_name=package_name, login=login)
        with open(os.path.join(package_dir, "ya.make"), "w") as f:
            f.write(src)

    # Move sources to sandbox/projects
    shutil.move(package_dir, task_dir)

    # Add PEERDIR and RECURSE

    ya_make_path = os.path.join(task_dir, "ya.make")
    add_to_macro(ya_make_path, "PEERDIR", "sandbox/projects/tutorial/{}".format(package_name))
    add_to_macro(ya_make_path, "RECURSE", package_name)

    task_name = os.path.basename(package_name).upper()
    print("New task type {} has been succesfully created at `{}`".format(task_name, os.path.relpath(destination)))


if __name__ == "__main__":
    main()
