#!/usr/bin/env python

from __future__ import print_function

import sys
if sys.version_info[:2] != (2, 7):
    print("This script requires python version 2.7", file=sys.stderr)
    sys.exit(1)

import os
import re
import copy
import json
import getpass
import logging
import argparse
import urlparse
import operator as op
import itertools as it
import contextlib
# noinspection PyUnresolvedReferences
import progressbar as pb

sandbox_dir = os.path.dirname(os.path.dirname(os.path.realpath(__file__)))
sys.path = [os.path.dirname(sandbox_dir), sandbox_dir] + sys.path

from sandbox import common
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
import sandbox.common.types.resource as ctr


MAX_OPEN_FILES = 1000  #: Amount of files can be opened simultaneously


def handle_files(args):
    ret = {}
    cwd = os.curdir
    unrel_re = re.compile(r"^(\.+/?)+")
    for root, _, files in it.chain.from_iterable(it.imap(
        lambda f: os.walk(f) if os.path.isdir(f) else ((os.path.dirname(f), tuple(), (os.path.basename(f),)),),
        args
    )):
        for subf in files:
            fname = os.path.join(root, subf)
            rpath = os.path.realpath(fname)
            if rpath not in ret:
                fh = open(fname, "rb")
                fh.seek(0, os.SEEK_END)
                ret[rpath] = common.upload.HTTPHandle.FileMeta(
                    fh if len(ret) < MAX_OPEN_FILES else rpath, fh.tell(), unrel_re.sub("", os.path.relpath(fname, cwd))
                )
                fh.seek(0)

    return sorted(ret.itervalues(), key=op.itemgetter(2))


def handle_args():
    parsed_url = list(urlparse.urlparse(common.rest.Client.DEFAULT_BASE_URL))
    parsed_url[0] = "https"    # scheme
    parsed_url[2] = ""         # path
    default_url = urlparse.urlunparse(parsed_url)

    parsed_url[0] = "http"     # scheme
    parsed_url[1] = ".".join(["proxy", parsed_url[1]])
    default_proxy_url = urlparse.urlunparse(parsed_url)

    parser = argparse.ArgumentParser(
        formatter_class=lambda *args, **kwargs: argparse.ArgumentDefaultsHelpFormatter(*args, width=120, **kwargs),
        description="Sandbox data upload tool."
    )
    parser.add_argument(
        "-v", "--verbosity",
        action="count", help="Increase output verbosity"
    )
    parser.add_argument(
        "-q", "--quiet",
        action="store_true", help="Silent or quiet mode. Don't show progress meter and runtime stages"
    )
    parser.add_argument(
        "-I", "--noninteractive",
        default=not sys.stdout.isatty(),
        action="store_true", help="Do try to interactively ask authorization information"
    )
    parser.add_argument(
        "-D", "--dump",
        default="", type=str,
        help="Dump created resource metadata in JSON format to the file specified (or STDOUT for '-')"

    )
    parser.add_argument(
        "-t", "--type",
        default="OTHER_RESOURCE", type=str, help="Resource type to register"
    )
    parser.add_argument(
        "-d", "--description",
        default="", type=str, help="Resource description to set"
    )
    parser.add_argument(
        "-A", "--attribute",
        type=str, nargs='*', help="Additional resource attribute(s) to set in form `key=value`"
    )
    parser.add_argument(
        "-a", "--arch",
        default=ctm.OSFamily.ANY, type=str, choices=list(ctm.OSFamily), help="Resource architecture to register"
    )
    parser.add_argument(
        "-U", "--url",
        default=default_url, type=str, help="Sandbox main URL to communicate with"
    )
    parser.add_argument(
        "-P", "--proxy-url",
        default=default_proxy_url, type=str, help="Sandbox proxy URL to upload the actual data with"
    )
    parser.add_argument(
        "-w", "--call-wait",
        default=None, type=int, help="Total response wait time limit per RPC call in seconds. No limit by default"
    )
    parser.add_argument(
        "-u", "--username",
        default=None, type=str, help="User name to request OAuth token for. Uses current system user name by default"
    )
    parser.add_argument(
        "-o", "--owner",
        default=None, type=str, help="Task/resource owner to be set. Uses user name if not specified"
    )
    parser.add_argument(
        "-k", "--private-key",
        default=None, type=str,
        help="Use private key file specified instead of default location and SSH agent to authenticate"
    )
    parser.add_argument(
        "-T", "--oauth-token",
        default=None, type=str,
        help="OAuth token to authenticate instead of SSH key or interactive mode. "
        "In case of '-' specified, the token will be taken from 'OAUTH_TOKEN' environment variable"
    )
    parser.add_argument(
        "-s", "--stream-name",
        default="stream.txt", type=str,
        help="File name to be created in case of content provided via STDIN stream"
    )
    parser.add_argument(
        "-l", "--stream-limit",
        default="1G", type=str,
        help="Stream limit in case of reading content from STDIN instead of file(s) from positional arguments. "
        "Possible suffixes are K, M, and G (powers of 1024)"
    )
    parser.add_argument(
        "-M", "--mds",
        default=None, action="store_true",
        help="Upload to MDS"
    )
    parser.add_argument(
        "-N", "--no-auth",
        default=None, action="store_true",
        help="No authorization"
    )
    parser.add_argument(
        "files", metavar="FILE", nargs="+",
        help="File(s)/directory(ies) list to upload. The script will read STDIN in case '-' file specified"
    )
    args = parser.parse_args()
    args.url = (args.url or default_url).rstrip("/")
    args.username = args.username or getpass.getuser()
    args.oauth_token = os.environ[ctm.OAUTH_TOKEN_ENV_NAME] if args.oauth_token == "-" else args.oauth_token
    args.stream_limit = common.format.str2size(args.stream_limit)
    if "-" in args.files:
        if len(args.files) != 1:
            raise ValueError("It is only one STDIN stream allowed in positional arguments")
        args.files = None
    return args


def main():
    args = handle_args()
    cz = common.console.AnsiColorizer()
    # File handle to output verbose information like progress meter and so on.
    pgfh = open(os.devnull, "wb") if args.quiet or (args.verbosity or 0) > 0 else sys.stderr
    version = re.search(r"upload\.(\d+)", sandbox_dir)
    logfile = common.log.script_log(
        args.verbosity, os.path.dirname(os.path.dirname(sandbox_dir)) if version else sandbox_dir, "upload.log"
    )
    version = version and version.group(1) or "UNKNOWN"
    logging.getLogger(__name__).info("Sandbox upload script v%s started.", version)
    print(cz.black("Sandbox upload script v{} at '{}', log at '{}'".format(version, sandbox_dir, logfile)), file=pgfh)

    task_palette = {
        k: cz.colorize(k, c)
        for c, k in it.chain.from_iterable((
            zip(it.repeat("green"), ctt.Status.Group.FINISH),
            zip(it.repeat("blue"), ctt.Status.Group.EXECUTE),
            zip(it.repeat("yellow"), ctt.Status.Group.BREAK),
            zip(it.repeat("white"), ctt.Status.Group.QUEUE),
            zip(it.repeat("black"), ctt.Status.Group.DRAFT),
        ))
    }
    task_palette.update({
        k: cz.colorize(c, k) for k, c in zip(("black", "red"), (ctt.Status.DELETED, ctt.Status.FAILURE))
    })
    resource_palette = {
        k: cz.colorize(k, c)
        for c, k in it.chain.from_iterable((
            zip(
                ("white", "green", "red", "black"),
                (ctr.State.NOT_READY, ctr.State.READY, ctr.State.BROKEN, ctr.State.DELETED)
            ),
        ))
    }

    token = common.console.Token(args.url, args.username, not args.noninteractive, pgfh)
    handle_class = common.upload.MDSHandle if args.mds else common.upload.HTTPHandle
    h = handle_class(
        common.upload.HTTPHandle.ResourceMeta(
            args.type, args.arch, args.owner or args.username, args.description,
            {
                k.strip(): v.strip()
                for k, v in (a.split("=") for a in common.itertools.chain(*(_.split(",") for _ in args.attribute)))
            } if args.attribute else None,
            release_to_yd=False
        ),
        (
            common.auth.NoAuth()
            if args.no_auth else
            (
                args.oauth_token
                if args.oauth_token and token.check(args.oauth_token) else
                token(args.private_key)
            )
        ),
        args.url,
        args.proxy_url,
        args.call_wait,
        *(
            handle_files(args.files)
            if args.files else
            [common.upload.HTTPHandle.StreamMeta(sys.stdin, args.stream_limit, args.stream_name)]
        )
    )
    _U = ctm.Upload
    state, last_state, last_state_copy = (None, ) * 3

    cop = common.console.Operation(pgfh)
    for state in h():
        if last_state != state:  # Last operation stopped.
            if isinstance(last_state, _U.Check):
                cop.long.intermediate(
                    "{} file(s) found totally for {}."
                    .format(last_state.amount, common.format.size2str(last_state.size))
                )
        else:  # Operation continues.
            if isinstance(state, _U.Prepare):
                if last_state_copy.task_id != state.task_id:
                    link = "{}/task/{}".format(args.url, state.task_id)
                    cop.long.intermediate(
                        "Task #{} created: {}".format(cz.white(state.task_id), cz.blue(link))
                    )
                if last_state_copy.resource_id != state.resource_id:
                    link = "{}/resource/{}".format(args.url, state.resource_id)
                    cop.long.intermediate(
                        "Resource #{} registered: {}".format(cz.white(state.resource_id), cz.blue(link))
                    )
            if isinstance(state, _U.DataTransfer):
                cop.pbar.update(state.done)
            if isinstance(state, _U.ShareResource):
                cop.long.intermediate(
                    "Resource currently is in '{}' state.".format(resource_palette[state.resource_state]),
                    cr=True
                )
            elif isinstance(state, _U.Share):
                cop.long.intermediate(
                    "Task currently is in '{}' state.    ".format(task_palette[state.task_state]),
                    cr=True
                )

        if last_state != state:  # Start new operation report.
            cop.finish(last_state.done if isinstance(last_state, _U.DataTransfer) else None)

            if isinstance(state, _U.Check):
                cop.long = "Calculating total files size"

            if isinstance(state, _U.Prepare):
                cop.long = "Preparing upload task"

            if isinstance(state, _U.DataTransfer):  # Prepare started
                size_label = pb.FormatLabel("%(cursize)s/%(maxsize)s")
                size_label.mapping["cursize"] = ("value", lambda x: common.format.size2str(x))
                size_label.mapping["maxsize"] = ("max_value", lambda x: common.format.size2str(x))
                if args.files:
                    cop.pbar = (
                        "", state.total,
                        [
                            "Uploading ",
                            pb.Bar(), " ",
                            pb.Percentage(), " | ",
                            size_label, " | ",
                            pb.Timer(), " | ",
                            pb.ETA(), " |",
                            pb.FileTransferSpeed()
                        ],
                    )
                else:
                    cop.pbar = (
                        "", state.total,
                        [
                            "[", pb.AnimatedMarker(), "] ",
                            "Uploading ",
                            size_label, " | ",
                            pb.Timer(), " | ",
                            pb.FileTransferSpeed()
                        ],
                    )

            if isinstance(state, _U.Share):
                cop.long = "Sharing uploaded data"

        last_state = state
        last_state_copy = copy.deepcopy(state)

    if isinstance(state, _U.Share):
        if isinstance(state, _U.ShareResource):
            cop.long.intermediate(
                "Resource currently is in '{}' state.    ".format(resource_palette[state.resource_state])
            )
            cop.long.intermediate(
                "Resource download link: {}".format(cz.blue(state.meta["mds"]["url"]))
            )
        else:
            cop.long.intermediate(
                "Task currently is in '{}' state.    ".format(task_palette[state.task_state])
            )
            cop.long.intermediate(
                "Resource download link: {}".format(
                    cz.blue("/".join([args.proxy_url.rstrip("/"), str(state.meta["id"])]))
                )
            )
        msg = None
        if state.skynet_id:
            msg = "Skybone ID is {}".format(cz.white(state.skynet_id))
        if state.md5sum:
            if msg:
                msg += ", "
            msg += "MD5 checksum is {}".format(cz.white(state.md5sum))
        if msg:
            cop.long.intermediate(msg)
    cop.finish()

    if args.dump:
        with (open(args.dump, "w") if args.dump != "-" else contextlib.closing(sys.stdout)) as fh:
            json.dump(state.meta, fh, indent=4, sort_keys=True, separators=(",", ": "))
            print("", file=fh)
    elif args.quiet:
        print(state.meta["id"])
    return 0


if __name__ == "__main__":
    try:
        sys.exit(main())
    except KeyboardInterrupt:
        print("Program interrupted by user.", file=sys.stderr)
        sys.exit(2)
