#!/usr/bin/env bash
set -e
set -o pipefail

# Author: mvel@

runtime_data="/home/$USER/work/tmp/sandbox_runtime_data"

opt=" --runtime $runtime_data "

myself="$(readlink -f "$0")"
sandbox_dir=$(dirname "$myself")/../../
cd $sandbox_dir

chmod +x sandbox
mkdir -p $runtime_data

sb_echo() {
    echo -e "[\x1b[1;36mSandboxUberManager\x1b[0m] $1"
}

sb_warning() {
    echo -e "[\x1b[1;31mSandboxUberManager\x1b[0m] $1"
}

sb_check_config() {
    if ! ./sandbox info $opt ; then
        sb_echo "Seems Sandbox is not configured. Setting up"
        ./sandbox setup $opt -p 33666
        sb_echo "Sandbox setup successful"
    fi
}

sb_is_started() {
    if ./sandbox info $opt 2>&1 | grep -q 'is launched' ; then
        return 0
    fi
    return 1
}

sb_start() {
    if ! sb_is_started ; then
        sb_echo "Seems Sandbox is not started. Starting..."
        ./sandbox start $opt
        sb_echo "Sandbox start successful"
    else
        sb_echo "Sandbox was already started"
    fi
}

sb_test_tasks() {
    ./sandbox test_tasks --quiet && sb_echo "Task tests are OK"
    ( cd ../sandbox-tasks/ ; find . -name '__pycache__' | xargs rm -rf ; cd - )
    rm -f ../sandbox-tasks/svn_*txt*
    rm -f ./svn_*txt*
    rm -f ./arcadia_*txt*
    rm -f ./ssh_*txt*
    sb_echo "Garbage cleanup complete"
}

sb_stop() {
    if  sb_is_started ; then
        sb_echo "Seems Sandbox started. Stopping..."
        ./sandbox stop $opt
        if ls -la $runtime_data/run/*.pid ; then
            sb_warning "Some PID files are still not removed. Run $0 cleanpid to remove them"
        fi
        sb_echo "Sandbox stop successful"
    else
        sb_echo "Sandbox was already stopped"
    fi
}

sb_cleanpid() {
    rm -f $runtime_data/run/*.pid
}

if [ -z "$1" ] ; then
    sb_echo "Usage: $0 [start|stop|cleanpid|clean_db|init|tasks|*] <options>"
    sb_echo "       Any command not in the list will be passed 'as is' to ./sandbox"
    exit 1
fi

sb_echo "Runtime data is located at '$runtime_data'"

if [ "$1" = "start" ] ; then
    sb_check_config
    sb_start
elif [ "$1" = "stop" ] ; then
    sb_check_config
    sb_stop
elif [ "$1" = "cleanpid" ] ; then
    sb_check_config
    sb_cleanpid
elif [ "$1" = "clean_db" ] ; then
    ./sandbox "$@"
elif [ "$1" = "init" ] ; then
    sb_stop
    chmod +w -R $runtime_data
    rm -rf $runtime_data
    mkdir -p $runtime_data
    sb_check_config
    sb_start
elif [ "$1" = "tasks" ] ; then
    sb_check_config
    sb_test_tasks
else
    # just proxy request to script
    ./sandbox "$@" --config "$runtime_data/configs/settings.yaml"
fi
