#!/skynet/python/bin/python
# -*- coding: utf-8 -*-
""""
Create resource from rbtorrent or from file (directory)
Inspired by some script from pg@ junk and ya-spe@ OAuth

Author: mvel@, lagrunge@
"""

from __future__ import print_function

import argparse
import os
import sys
import xmlrpclib
import getpass
import subprocess


def parse_options():
    parser = argparse.ArgumentParser(formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument('-t', '--target',
                        dest='target',
                        required=True,
                        help='What to upload. Filename or rbtorrent'
                        )
    parser.add_argument('--type',
                        dest='resource_type',
                        required=True,
                        # default='UKROP_FAVICON_DATABASE',
                        help='Resource type'
                        )
    parser.add_argument('--ttype',
                        dest='task_type',
                        default='REMOTE_COPY_RESOURCE',
                        help='Task type'
                        )
    parser.add_argument('-d', '--description',
                        dest='resource_description',
                        help='Resource description'
                        )
    parser.add_argument('-n', '--name',
                        dest='resource_name',
                        help='Resource name',
                        )
    parser.add_argument('-a', '--arch',
                        dest='architecture',
                        default='any',
                        help='Architecture'
                        )
    parser.add_argument('--user',
                        dest='sandbox_user',
                        default=os.getenv('SANDBOX_OAUTH_USER', getpass.getuser()),
                        help='Sandbox user. Will use SANDBOX_OAUTH_USER environment variable if'
                             ' available or username if not'
                        )
    parser.add_argument('--token',
                        dest='sandbox_token',
                        default=os.getenv('SANDBOX_OAUTH_TOKEN'),
                        help='Sandox OAuth token. Will use SANDBOX_OAUTH_TOKEN if not set.'
                             ' See https://nda.ya.ru/3QTU5c for details'
                        )
    return parser.parse_args()


# thanks ya-spe@ for this code
class OAuthTransport(xmlrpclib.Transport):
    """
        Wrapper for XMLRPC transport appending OAuth header
    """

    def __init__(self, token):
        xmlrpclib.Transport.__init__(self)
        self.token = token

    def send_request(self, connection, handler, request_body):
        xmlrpclib.Transport.send_request(self, connection, handler, request_body)
        if self.token:
            connection.putheader("Authorization", 'OAuth {}'.format(self.token))


def create_task(
        server,
        task_type,
        description,
        ctx,
        arch=None,
        model=None,
):
    """
        Create Sandbox task using XMLRPC
    """
    user = os.getenv('SANDBOX_OAUTH_USER', getpass.getuser())
    print('User detected: ' + user, file=sys.stderr)
    task = {
        'type_name': task_type,
        'author': user,
        'owner': user,
        'descr': description,
        'ctx': ctx,
        'priority': ('USER', 'NORMAL'),
    }

    if arch:
        task['arch'] = arch

    if model:
        task['model'] = model

    return server.create_task(task)


def upload_task(server, description, ctx, arch=None):
    """
        Create uploading task
    """
    return create_task(server, ctx['task_type'], description, ctx, arch=arch)


def upload(server, resource_id, resource_type, description, arch='any', resource_name='RESOURCE', **kwargs):
    """
        Upload file to Sandbox
    """

    if 'task_type' not in kwargs:
        task_type = 'REMOTE_COPY_RESOURCE'
    else:
        task_type = kwargs.get('task_type')

    return upload_task(
        server=server,
        description=description,
        ctx={
            'task_type': task_type,
            'resource_type': resource_type,
            'remote_file_name': resource_id,
            'remote_file_protocol': 'skynet',
            'created_resource_name': resource_name,
            'arch': arch,
        }
    )


def _main(args):
    """
        Gravicappa
    """
    rbtorrent = None
    resource_name = 'RESOURCE'

    if os.path.isfile(args.target):  # if target was a file
        file_to_share = os.path.realpath(args.target)
        file_to_share_dir = os.path.dirname(file_to_share)

        # given parameter is a file, create a torrent from it
        resource_name = os.path.basename(file_to_share)

        print("Sharing '{}' with SkyNet...".format(file_to_share), file=sys.stderr)
        sky_command = ["sky", "share", '-d', file_to_share_dir, file_to_share]
        try:
            proc = subprocess.Popen(
                sky_command,
                stdin=subprocess.PIPE,
                stdout=subprocess.PIPE,
            )
            out, err = proc.communicate()
            if proc.returncode != 0:
                print("Failed to share, SkyNet answer was:", file=sys.stderr)
                print(err, file=sys.stderr)
                sys.exit(1)
        except OSError as e:
            if e.errno == 2:
                print(
                    'ERROR: SkyNet executable is not found. Failed command details:\n' +
                    '   ' + ' '.join(sky_command) + '\n' +
                    'Please check you have SkyNet (`sky` command) in your system\n',
                    file=sys.stderr
                )
                sys.exit(1)
            raise

        rbtorrent = out.strip()
    elif args.target.startswith('rbtorrent:'):  # if target is a file
        rbtorrent = args.target
    else:
        print('Invalid torrent or file specified "{}"'.format(args.target), file=sys.stderr)
        sys.exit(1)

    if not args.sandbox_token:
        print(
            'WARNING: OAuth token was not set, task priority will be very low. '
            'See https://nda.ya.ru/3QTU5c for details',
            file=sys.stderr
        )

    server = xmlrpclib.ServerProxy(
        'https://sandbox.yandex-team.ru/sandbox/xmlrpc',
        allow_none=True,
        transport=OAuthTransport(args.sandbox_token),
    )

    if args.task_type is not None:
        task_type = args.task_type

    if args.resource_name is not None:
        resource_name = args.resource_name

    description = resource_name + ' made from ' + rbtorrent
    if args.resource_description is not None:
        description = args.resource_description

    result = upload(
        server=server,
        resource_id=rbtorrent,
        resource_type=args.resource_type,
        description=description,
        resource_name=resource_name,
        arch=args.architecture,
        task_type=task_type
    )

    print(
        'https://sandbox.yandex-team.ru/sandbox/tasks/view?task_id={}'.format(result),
        file=sys.stdout
    )


if '__main__' == __name__:
    args = parse_options()
    _main(args)
