from __future__ import absolute_import

import os
import abc
import random
import logging
import itertools as it
import subprocess as sp

import six
from six.moves.urllib import parse as urlparse

from sandbox.common import fs
from sandbox.common import share
from sandbox.common import config
from sandbox.common import errors
from sandbox.common import itertools
from sandbox.common.types import misc as ctm

from .vcs import svn


class RemoteCopyMeta(abc.ABCMeta):
    registered_types = []

    def __new__(mcs, name, bases, cls_dict):
        cls = super(RemoteCopyMeta, mcs).__new__(mcs, name, bases, cls_dict)
        if len(cls.__mro__) > 2:
            mcs.registered_types.append(cls)
        return cls


class RemoteCopy(six.with_metaclass(RemoteCopyMeta, object)):
    DEFAULT_TIMEOUT = 3 * 60 * 60
    DEFAULT_PREFIX = "remote_copy"

    class ProcessLog(object):
        def __init__(self, log_dir, prefix=None):
            self.__log_dir = log_dir
            self.__prefix = prefix or RemoteCopy.DEFAULT_PREFIX
            self.__stdout = None
            self.__stderr = None

        @property
        def stdout(self):
            return self.__stdout

        @property
        def stderr(self):
            return self.__stderr

        @staticmethod
        def __unique_filename(filename):
            i = 1
            while True:
                unique_filename = "{}.{}".format(filename, i)
                if not os.path.exists(unique_filename):
                    return unique_filename
                i += 1

        def __enter__(self):
            if self.__log_dir:
                log_dir = fs.make_folder(self.__log_dir)
                self.__stdout = open(
                    self.__unique_filename(os.path.join(log_dir, "{}.out.txt".format(self.__prefix))),
                    "w"
                )
                self.__stderr = open(
                    self.__unique_filename(os.path.join(log_dir, "{}.err.txt".format(self.__prefix))),
                    "w"
                )
            return self

        def __exit__(self, *_):
            if not self.__log_dir:
                return
            self.__stdout.close()
            self.__stderr.close()

    def __new__(cls, src, dst, **kws):
        if len(cls.__mro__) > 2:
            return object.__new__(cls)
        # noinspection PyUnresolvedReferences
        parsed_url = urlparse.urlparse(src)
        compatibles = list(it.chain(
            (subcls for subcls in type(cls).registered_types if subcls.compatible(src, parsed_url)),
            [RemoteCopySkynet]
        ))
        return object.__new__(compatibles[0])

    def __init__(self, src, dst, log_dir=None):
        logging.debug("RemoteCopy: %s -> %s", src, dst)
        self._src = src
        self._dst = dst
        self._log_dir = log_dir

    @classmethod
    def _wait_process(cls, process, timeout):

        def checker():
            return process.poll() is not None

        if itertools.progressive_waiter(0.01, 1, timeout, checker)[0]:
            return cls._check_retcode(process)
        try:
            process.terminate()
            if not itertools.progressive_waiter(0.01, 1, 5, checker)[0]:
                process.kill()
        except OSError:
            pass
        raise errors.SubprocessError("Process was interrupted by timeout: {}".format(timeout))

    @staticmethod
    def _opts(opts):
        for name, value in six.iteritems(opts):
            name = "--{}".format(name.replace("_", "-"))
            if value is True:
                yield name
            elif value is False or value is None:
                continue
            elif isinstance(value, list):
                for v in value:
                    yield name
                    yield str(v)
            else:
                yield name
                yield str(value)

    @staticmethod
    def _check_retcode(process, wait=False):
        retcode = process.wait() if wait else process.returncode
        if retcode:
            raise errors.SubprocessError("Process exited with non-zero return code {}".format(retcode))

    @abc.abstractmethod
    def compatible(self, src, parsed_url):
        pass

    @abc.abstractmethod
    def __call__(self, *args, **kwargs):
        pass


class RemoteCopyRcp(RemoteCopy):
    SCHEME = "rcp"

    @classmethod
    def compatible(cls, src, parsed_url):
        return parsed_url.scheme == cls.SCHEME

    def __call__(self, timeout=RemoteCopy.DEFAULT_TIMEOUT, **kws):
        cmd = it.chain((self.SCHEME, "-r"), self._opts(kws), (self._src, self._dst))
        with self.ProcessLog(self._log_dir) as plog:
            # noinspection PyTypeChecker
            process = sp.Popen(cmd, close_fds=True, stdout=plog.stdout, stderr=plog.stderr)
            self._wait_process(process, timeout=timeout)


class RemoteCopyScp(RemoteCopyRcp):
    SCHEME = "scp"


class RemoteCopyHttp(RemoteCopy):
    @classmethod
    def compatible(cls, src, parsed_url):
        return parsed_url.scheme in ("http", "https")

    def __call__(self, timeout=60, headers=None, show_error=True, **kws):
        headers = (
            it.chain.from_iterable(("--header", "{}: {}".format(k, v)) for k, v in six.iteritems(headers))
            if headers else
            ()
        )
        cmd = list(
            it.chain(
                ("curl", "-f", "-L", "-m", str(timeout)),
                headers,
                self._opts(kws),
                ("-o", self._dst, self._src)
            )
        )
        with self.ProcessLog(self._log_dir) as plog:
            logging.debug("Fetching data. Command line is %r", " ".join(cmd))
            self._check_retcode(sp.Popen(cmd, close_fds=True, stdout=plog.stdout, stderr=plog.stderr), True)


class RemoteCopyRsync(RemoteCopy):
    DEFAULT_ARGS = {
        "checksum": True,                       # skip based on checksum, not mod-time & size
        "out_format": "[%t] %i %n%L",           # set custom output format
        "progress": True,                       # show progress during transfer
        "recursive": True,                      # recurse into directories
        "inplace": True,                        # update destination files in-place
        "partial": True,                        # keep partially transferred files
        "links": True,                          # copy symlinks as symlinks
        "perms": True,                          # preserve permissions
        "group": True,                          # preserve group
        "chmod": "+w",                          # affect file and/or directory permissions
        "contimeout": 60,                       # set daemon connection timeout in seconds
        "bwlimit": 75000,                       # limit I/O bandwidth; KBytes per second
        "timeout": RemoteCopy.DEFAULT_TIMEOUT,  # set I/O timeout in seconds
        "exclude": None,                        # exclude files matching pattern
        "times": False                          # preserve modification times
    }

    @classmethod
    def compatible(cls, src, parsed_url):
        return parsed_url.scheme == "rsync"

    def __call__(self, resource_id=None, **kws):
        for name, value in six.iteritems(self.DEFAULT_ARGS):
            if (
                name == "contimeout" and
                config.Registry().this.system.family not in ctm.OSFamily.Group.OSX
            ):
                value = None
            kws.setdefault(name, value)
        try:
            # This actually a hack to load skynet.copier's egg.
            # In case it will be broken, the code will "silently" fallback to non-fasbonized mode.
            # noinspection PyUnresolvedReferences
            import api.copier
            api.copier.Copier()
            try:
                import ya.skynet.services.copier.client.utils as copier_tools
            except ImportError:
                # noinspection PyUnresolvedReferences
                import ya.skynet.services.skybone.client.utils as copier_tools
            urls = [self._src, copier_tools.fastbonizeURL(self._src)]
        except Exception as ex:
            logging.error("Unable to load skynet.copier's utility module: %s", str(ex))
            urls = [self._src, self._src]
        if urls[0] == urls[1]:
            urls.pop()

        while urls:
            url = urls.pop()
            log_suffix = str(resource_id) if resource_id else "{:#x}".format(random.randrange(0xFFFFFFFF))
            try:
                cmd = list(it.chain(("rsync", "-vvv"), self._opts(kws), (url, self._dst)))
                logging.debug("Fetching data from %r. Command line is %r", url, " ".join(cmd))
                with self.ProcessLog(self._log_dir, "{}.{}".format(self.DEFAULT_PREFIX, log_suffix)) as plog:
                    process = sp.Popen(cmd, close_fds=True, stdout=plog.stdout, stderr=plog.stderr)
                    timeout = kws.get("timeout")
                    self._wait_process(process, timeout=timeout)
                if plog.stdout:
                    with open(plog.stdout.name) as f:
                        logging.info(
                            "Rsync #%s summary:\n%s",
                            log_suffix,
                            "\n".join(list(fs.tail(f, 4))[:2])
                        )
                    os.remove(plog.stdout.name)
                break
            except errors.SubprocessError:
                if not urls:
                    raise
                logging.exception("Error fetching data via '%s'", url)


class RemoteCopySvn(RemoteCopy):
    @classmethod
    def compatible(cls, src, parsed_url):
        return parsed_url.scheme == "svn+ssh" and parsed_url.netloc != "arcadia.yandex.ru"

    def __call__(self, **kws):
        cmd = it.chain(("svn", "export"), self._opts(kws), (self._src, self._dst))
        with self.ProcessLog(self._log_dir) as plog:
            # noinspection PyTypeChecker
            process = sp.Popen(cmd, close_fds=True, stdout=plog.stdout, stderr=plog.stderr)
            return process.wait()


class RemoteCopyArcadia(RemoteCopy):
    @classmethod
    def compatible(cls, src, parsed_url):
        return any((
            parsed_url.scheme == "arcadia",
            parsed_url.scheme == "svn+ssh" and parsed_url.netloc == "arcadia.yandex.ru"
        ))

    def __call__(self, **kws):
        svn.Arcadia.export(self._src, self._dst)


class RemoteCopySkynet(RemoteCopy):
    @classmethod
    def compatible(cls, src, parsed_url):
        # see SANDBOX-8861
        return src.startswith('rbtorrent:')

    def __call__(self, unstable=None, exclude=None, timeout=RemoteCopy.DEFAULT_TIMEOUT, fallback_to_bb=False, **kws):
        # noinspection PyUnresolvedReferences
        if self.compatible(self._src, urlparse.urlparse(self._src)):
            share.skynet_get(self._src, self._dst, timeout, fallback_to_bb)
        else:
            host, _, srcdir = self._src.partition(":")
            share.skynet_copy(host, srcdir, self._dst, unstable=unstable, exclude=exclude)
