""" Environments classes used in Sandbox tasks """

from __future__ import absolute_import

import re
import os
import sys
import abc
import csv
import time
import errno
import shlex
import shutil
import random
import logging
import numbers
import tarfile
import tempfile
import textwrap
import datetime as dt
import platform
import importlib
import contextlib
import compileall
import subprocess as sp
import collections
import pkg_resources
import distutils.util

import six

import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
import sandbox.common.types.client as ctc
import sandbox.common.types.resource as ctr
from sandbox.common import os as common_os
from sandbox.common import config as common_config
from sandbox.common import errors as common_errors
from sandbox.common import context as common_context
from sandbox.common import patterns as common_patterns
from sandbox.common import platform as common_platform
from sandbox.common import itertools as common_itertools
from sandbox.common import threading as common_threading

from . import paths
from . import legacy
from . import helpers
from . import task


class FileSizeMetadata(object):

    metadata_file = ".metadata"

    def __init__(self, path, metadata_file=None):
        self.path = path
        if metadata_file:
            self.metadata_file = metadata_file

        self.full_path = os.path.join(path, self.metadata_file)

    def _write(self, metadata):
        with common_threading.FLock(self.full_path):
            with open(self.full_path, "w") as f:
                writer = csv.writer(f, delimiter="\t")
                for row in metadata:
                    writer.writerow(row)

    def _load(self):
        with common_threading.FLock(self.full_path):
            with open(self.full_path) as f:
                return [(fname, int(size)) for fname, size in filter(None, csv.reader(f, delimiter="\t"))]

    def _iter_files(self):
        destination = self.path
        for root, dirs, files in os.walk(destination):
            for name in files:
                full_path = os.path.join(root, name)
                if full_path.startswith(self.full_path):
                    # ignore metadata file itself and .metadata.lock_ in case of Windows
                    continue
                file_size = os.path.getsize(full_path) if os.path.isfile(full_path) else 0
                yield os.path.relpath(full_path, destination), file_size

    @property
    def exists(self):
        return os.path.exists(self.full_path)

    @property
    def mtime(self):
        if self.exists:
            return os.path.getmtime(self.full_path)

    def outdated(self, delta, now=None):
        if not self.exists:
            return True

        if now is None:
            now = dt.datetime.utcnow()
        elif isinstance(now, numbers.Real):
            now = dt.datetime.utcfromtimestamp(now)
        else:
            raise ValueError("Bad now param. Should be datetime or int or float")

        return (now - dt.datetime.utcfromtimestamp(self.mtime)) > delta

    def touch(self):
        if self.exists:
            os.utime(self.full_path, None)
        else:
            paths.make_folder(self.path)
            self._write([])

    def store(self):
        if self.exists:
            os.unlink(self.full_path)
        self._write(self._iter_files())

    def check_files(self):
        real = dict(self._iter_files())
        saved = dict(self._load())
        real_set = frozenset(real)
        saved_set = frozenset(saved)

        absent = saved_set - real_set
        extra = real_set - saved_set

        different = set()
        for f in saved:
            if f not in absent:
                real_size = real[f]
                stored_size = saved[f]
                if stored_size != real_size:
                    different.add((f, (stored_size, real_size)))

        return absent, extra, different


class PipError(Exception):
    pass


class VirtualEnvironment(object):
    """
    VirtualEnv wrapper.
    For execution tasks in virtualenv.

    Usage example:

    with VirtualEnvironment('/tmp') as venv:
        venv.pip('pytest==2.5.1 pytest-xdist==1.10 pytest-pep8==1.0.5')
        # or venv.pip('-r requirements.txt')
        suprocess.call([venv.executable, 'somescript.py'])
    """

    if platform.system() == "Windows":
        VIRTUALENV_EXE = "C:\\Python27\\Scripts\\virtualenv.exe"
    else:
        VIRTUALENV_EXE = "/skynet/python/bin/virtualenv"
    PYPI_URLS = ["https://pypi.yandex-team.ru/simple/"]

    def __init__(
        self, working_dir=".", use_system=False, do_not_remove=False,
        python_exe=None, venv_args=("-m", "virtualenv")
    ):
        """
        :param working_dir: directory in which will be created a directory with the virtual environment (venv)
        :param use_system: use --system-site-packages option of virtualenv
        :param do_not_remove: do not remove venv directory on exit
        :param python_exe: path to custom python executable. Should contain virtualenv module
        :param venv_args: args for custom python executable to invoke virtualenv.
        """
        self.__venv_dir = os.path.abspath(os.path.join(working_dir, "venv"))
        self.__use_system = use_system
        self.__do_not_remove = do_not_remove
        self.__pip_wrapper = os.path.join(self.__venv_dir, "pip_wrapper.py")
        self.__updated = False
        self.__custom_python = bool(python_exe)
        self.venv_python = None

        if python_exe:
            self.__virtualenv_args = ((python_exe,) + venv_args)
        else:
            self.__virtualenv_args = tuple(shlex.split(self.VIRTUALENV_EXE, posix="posix" in os.name))

        if platform.system() == "Windows":
            self.__executable = os.path.join(self.__venv_dir, "Scripts", "python.exe")
            self.__rm_cmd = ("cmd.exe", "/C", "rd", "/s", "/q")
        else:
            self.__executable = os.path.join(self.__venv_dir, "bin", "python")
            self.__rm_cmd = ("rm", "-rf")

        if python_exe and not os.path.isfile(python_exe):
            raise IOError("Virtualenv executable not found in {}".format(python_exe))
        if python_exe and os.path.abspath(python_exe).startswith("/skynet/python/"):
            raise RuntimeError("You should not use skynet python as custom python")

    @staticmethod
    def __run_cmd(cmd, extra_env=None, log_prefix=None):
        env = None
        if extra_env:
            env = os.environ.copy()
            env.update(extra_env)
        with helpers.ProcessLog(logger=log_prefix) as pl:
            helpers.process.subprocess.check_call(cmd, env=env, stdout=pl.stdout, stderr=sp.STDOUT)

    def __enter__(self):
        self.__pythonpath = os.environ.get("PYTHONPATH")
        self.__path = os.environ.get("PATH", "")
        if platform.system() == "Windows":
            python_path = "c:\\Python27\\"
            venv_bin = os.path.join(self.__venv_dir, "Scripts")
        else:
            python_path = "/skynet"
            venv_bin = os.path.join(self.__venv_dir, "bin")

        if not self.__custom_python:
            os.environ["PYTHONPATH"] = python_path
        SandboxEnvironment.update_os_env("PATH", venv_bin)
        self.__run_cmd(
            self.__virtualenv_args +
            (("--system-site-packages",) if self.__use_system else ()) +
            (self.__venv_dir,)
        )
        self.venv_python = self.__executable
        with open(self.__pip_wrapper, "w") as f:
            f.write(textwrap.dedent("""
                import importlib
                import sys
                import time
                import urllib2

                __do_open = urllib2.AbstractHTTPHandler.do_open
                def do_open(self, http_class, req):
                    for i in range(10):
                        try:
                            return __do_open(self, http_class, req)
                        except urllib2.URLError as exc:
                            if "[Errno 104]" in str(exc):
                                time.sleep(0.1)
                                continue
                            raise
                urllib2.AbstractHTTPHandler.do_open = do_open

                def pip_main_func():
                    for module_name in (
                        "pip._internal.cli.main",  # pip >= 20
                        "pip._internal.main",  # pip >= 19.3, < 20
                        "pip._internal",  # pip >= 10, < 19.3
                        "pip",  # pip < 10
                    ):
                        try:
                            return importlib.import_module(module_name).main
                        except (ImportError, AttributeError):
                            pass
                    raise NotImplementedError("This pip version is not supported")

                if __name__ == "__main__":
                    pip_main = pip_main_func()
                    sys.exit(pip_main(sys.argv[1:]))
            """))
        pydistutils_cfg = os.path.join(self.__venv_dir, ".pydistutils.cfg")
        with open(pydistutils_cfg, "w") as f:
            f.write(textwrap.dedent("""
                [easy_install]
                index_url = {}
            """.format(self.PYPI_URLS[0])))
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        if self.__pythonpath:
            os.environ["PYTHONPATH"] = self.__pythonpath
        else:
            os.environ.pop("PYTHONPATH", None)
        if self.__path:
            os.environ["PATH"] = self.__path
        else:
            os.environ.pop("PATH", None)
        if not self.__do_not_remove:
            self.__run_cmd(self.__rm_cmd + (self.__venv_dir,))

    @property
    def root_dir(self):
        """ Returns path to the virtual environment. """
        return self.__venv_dir

    @property
    def executable(self):
        """ Returns path to the virtual environment's bound interpreter executable. """
        if not self.__custom_python:
            self.fix_dylib_paths()
            self.fix_self_rpaths()
        return self.__executable

    @property
    def user_site(self):
        """ Returns user site-packages availability from the virtual environment. """

        code = (
            "import site; "
            "print(site.ENABLE_USER_SITE)"
        )

        proc = self.run_code(code=code)

        # Command returns 2 lines: first line with status and the second empty line.
        # Get 1st line ( [0] ) and remove excess symbols (trailing newlines, spaces, etc) ( strip() ).
        user_site_status = proc.communicate()[0].strip()

        # user_site_status contains string with 'True' or 'False' text. So, it can not be
        # transparently translated to bool in case of python strong typing.
        # Use special function for casting 'str' to 'bool'.
        return distutils.util.strtobool(user_site_status)

    @staticmethod
    def get_extra_build_variables(fix_rpath=True, use_cflags_ldflags=True):
        env = {}
        system_family = common_config.Registry().this.system.family

        if fix_rpath:
            if system_family in (ctm.OSFamily.LINUX, ctm.OSFamily.LINUX_ARM):
                # Reserve space for RPATH
                # Set rpath in libraries, analogue LDFLAGS="-Wl,-R/skynet/python/lib"
                env["LD_RUN_PATH"] = ":" * 256
                env["LIBRARY_PATH"] = "/skynet/python/lib"

        if use_cflags_ldflags and system_family in (
            ctm.OSFamily.LINUX, ctm.OSFamily.LINUX_ARM, ctm.OSFamily.OSX, ctm.OSFamily.OSX_ARM
        ):
            env["CFLAGS"] = "-I/usr/local/include"
            env["LDFLAGS"] = "-L/skynet/python/lib -L/usr/local/lib"

            if common_config.Registry().this.system.family in (ctm.OSFamily.LINUX, ctm.OSFamily.LINUX_ARM):
                env["LDFLAGS"] += " -Wl,-R/skynet/python/lib"

        return env

    def run_code(self, code):
        """
        Run given python code in virtual environment, a wrapper around `<python executable> -c "code"`.

        :param code: python code to run
        :param run_process_args: arguments for `process.run_process`, used to run code in virtualenvironment.
          Attention: `cmd` and `shell` arguments will be overridden.
        :return: a wrapper around running process
        :rtype: :py:class:`subprocess.Popen`
        """
        cmd_line = '{executable} -c "{code}"'.format(executable=self.executable, code=code)
        command = shlex.split(cmd_line)
        return sp.Popen(command, stdout=sp.PIPE, stderr=sp.PIPE)

    def get_module_version(self, module):
        """
        Get version of specific module

        :param module: name of module to get version
        :rtype: tuple
        :return: pkg_resources.parse_version call result ( None, when module is not installed yet ).
        """

        failure_message = "Failure"

        # Suppress any tracebacks from version get command
        code = textwrap.dedent("""\
            try:
                import {module}
                version = {module}.__version__
            except:
                version = '{failure_message}'

            print(version)
        """.format(module=module, failure_message=failure_message))

        proc = self.run_code(code=code)

        module_version_line = proc.communicate()[0].strip()

        if module_version_line == failure_message:
            module_version = None
        else:
            module_version = pkg_resources.parse_version(module_version_line)

        return module_version

    def pip(self, req_specs, extra_env=None, index_url=None, use_cflags_ldflags=True, log_prefix="venv_pip"):
        """ Installs given requirement into the virtual environment. """
        if not extra_env:
            extra_env = {}

        if not self.__custom_python:
            extra_env.update(
                self.get_extra_build_variables(
                    use_cflags_ldflags=use_cflags_ldflags
                )
            )

        pypi_urls = [index_url] + self.PYPI_URLS if index_url and index_url not in self.PYPI_URLS else self.PYPI_URLS

        for pypi_url in pypi_urls:
            try:
                self.__run_cmd(
                    (
                        self.__executable, "-us",
                        self.__pip_wrapper, "install",
                        "-i", pypi_url,
                        "--upgrade",
                    ) + tuple(shlex.split(req_specs, posix="posix" in os.name)),
                    extra_env=extra_env,
                    log_prefix=log_prefix,
                )
                break
            except helpers.process.subprocess.CalledProcessError:
                logging.exception("Error occurred")
        else:
            raise PipError("Can not install {}. See logs for more info.".format(req_specs))
        self.__updated = True

    @classmethod
    def _parse_requirement_file(cls, fname):
        import pip.req
        import pip.download

        pip_session = pip.download.PipSession()

        return set(str(req.req.key) for req in pip.req.parse_requirements(fname, session=pip_session))

    def check_requirements(self, check_pkgs):
        with tempfile.NamedTemporaryFile() as f:
            for line in check_pkgs.split(" "):
                f.write(line.strip() + "\n")
            f.flush()
            return self.check_requirements_from_file(f.name)

    def check_requirements_from_file(self, fname):
        required = self._parse_requirement_file(fname)

        cmd = [
            self.__executable,
            "-us",
            self.__pip_wrapper,
            "freeze"
        ]
        with tempfile.NamedTemporaryFile() as f, open(os.devnull, "wb") as devnull:
            try:
                sp.check_call(cmd, stderr=devnull, stdout=f.file)
            except sp.CalledProcessError as ex:
                raise common_errors.SubprocessError(str(ex))
            f.flush()
            available = self._parse_requirement_file(f.name)

        logging.debug("Check venv requirements: %s. Available: %s", required, available)

        missing = required - available
        if missing:
            err_msg = "Missing packages in venv: {}".format(list(missing))
            logging.error(err_msg)
            raise common_errors.TaskFailure(err_msg)

    @staticmethod
    def fix_rpaths(root_path, check_elf=False):
        def so_walker(path):
            for root, dirs, files in os.walk(path):
                for f in files:
                    if f.endswith(".so"):
                        yield os.path.join(root, f)

        for fname in so_walker(root_path):
            rel_path = os.path.relpath(fname, root_path)
            logging.info("Updating library '%s'", rel_path)
            try:
                sp.check_output(
                    ["/skynet/python/bin/chrpath", "--replace", "/skynet/python/lib", fname],
                    stderr=sp.STDOUT
                )
            except sp.CalledProcessError as ex:
                if ex.returncode == 2:
                    logging.warning("Lib '%s' have no RPATH", rel_path)
                else:
                    logging.error("Fail updating RPATH '%s': %s", rel_path, ex.output)
                return False

        if check_elf:
            def elf_walker(path):
                for root, dirs, files in os.walk(path):
                    for f in files:
                        if "." in f:  # In UNIX elf usually without extension
                            continue
                        yield os.path.join(root, f)

            devnull = open(os.devnull, "wb")
            rpath_regex = re.compile(r".*\(RPATH\)\s+Library rpath: \[(.*\])")
            for fname in elf_walker(root_path):
                rel_path = os.path.relpath(fname, root_path)
                try:
                    output = sp.check_output(["readelf", "-d", fname], stderr=devnull)
                except sp.CalledProcessError as ex:
                    if ex.returncode == 1:  # Not an ELF file
                        continue
                    raise

                for line in output.splitlines():
                    m = rpath_regex.match(line)
                    if m:
                        rpath = m.group(1)
                        if "/skynet/python/lib" in rpath:
                            logging.info("Check executable '%s': OK (%s)", rel_path, rpath)
                        else:
                            logging.warning(
                                "Check executable '%s': FAIL. Absent /skynet/python/lib in %s", rel_path, rpath
                            )
                        break
                else:
                    logging.warning("Check executable '%s': FAIL. No RPATH", rel_path)
        return True

    def fix_self_rpaths(self):
        if not (
            self.__updated and
            common_config.Registry().this.system.family in (ctm.OSFamily.LINUX, ctm.OSFamily.LINUX_ARM)
        ):
            return

        return self.fix_rpaths(self.root_dir)

    def fix_dylib_paths(self, additional=None):
        """ Point runtime linker to skynet python library's absolute path on OSX """
        if not (
            self.__updated and common_config.Registry().this.system.family in ctm.OSFamily.Group.OSX
        ):
            return
        logging.info("Post-processing virtual environment dynamic libraries.")
        # First of all, fix python executable
        python = os.path.join(self.root_dir, "bin", "python")
        pylib = "/skynet/python/lib/libpython2.7.dylib"
        logging.info("Updating python executable '%s'", python)
        sp.check_call([
            "install_name_tool", "-change",
            os.path.realpath(pylib), pylib,
            python
        ])
        update = common_itertools.chain(
            additional or [],
            *[
                [os.path.join(root, _) for _ in files if _.endswith(".so") or _.endswith(".dylib")]
                for root, dirs, files in os.walk(os.path.join(self.root_dir, "lib"))
            ]
        )
        mapping = (
            ("skynet/libpython2.7.dylib", pylib),
            ("/System/Library/Frameworks/Python.framework/Versions/2.7/Python", pylib),
        )
        for fname in update:
            logging.debug("Checking file '%s'", fname)
            try:
                otool = [_.strip() for _ in sp.check_output(["otool", "-L", fname]).splitlines()]
                for p1, p2 in mapping:
                    p1chk = p1 + " "
                    need_update = any(_.startswith(p1chk) for _ in otool)
                    if need_update:
                        logging.info("Updating path to %r->%r for dynamic library '%s'", p1, p2, fname)
                        sp.check_call(["install_name_tool", "-change", p1, p2, fname])
            except sp.CalledProcessError as ex:
                logging.error("Error processing file %r: %s", fname, ex)
        self.__updated = False

    def make_relocatable(self, log_prefix="venv"):
        """ Makes virtualenv relocatable (<venv_executable> --relocatable <venv_directory>) """
        cmd = self.__virtualenv_args + ("--relocatable", self.__venv_dir)
        self.__run_cmd(cmd, log_prefix=log_prefix)

    def pack_bundle(self, bundle_path, log_prefix="venv"):
        """ Packs the virtual environment as a bundle, which can be reused later on the similar platform. """
        try:
            if not self.__custom_python:
                self.fix_dylib_paths()
                self.fix_self_rpaths()
            compileall.compile_dir(os.path.join(self.__venv_dir, "lib"), force=True)
            # relocate virtualenv
            self.make_relocatable(log_prefix=log_prefix)
            with contextlib.closing(tarfile.open(bundle_path, "w:gz")) as tar_file:
                for fname in os.listdir(self.__venv_dir):
                    if fname in ("pip_wrapper.py", ".pydistutils.cfg"):
                        continue
                    tar_file.add(
                        os.path.join(self.__venv_dir, fname),
                        arcname=fname
                    )
        except Exception as e:
            raise common_errors.TaskFailure(e.message)


class OSEnv(common_patterns.Abstract):
    __slots__ = ("exe_path", "dll_path")
    __defs__ = ([], [])


class EnvironmentRegistrar(abc.ABCMeta):

    __environments = collections.defaultdict(list)

    def __new__(mcs, name, bases, namespace):
        cls = super(EnvironmentRegistrar, mcs).__new__(mcs, name, bases, namespace)
        mcs.__environments[cls.__name__].append(cls)
        return cls

    def __iter__(cls):
        return iter(cls.__environments)

    def get_environment_class(cls, name):
        if name not in cls.__environments:
            raise KeyError("Unknown environment {!r}".format(name))
        return cls.__environments.get(name, [None])[0]


class SandboxEnvironment(six.with_metaclass(EnvironmentRegistrar, object)):
    """
    Base class for various toolkits, which can be automatically deployed and set up at the host, where the task,
    which uses it, is executing.
    """

    #: Resource type to be used to search compatible resources
    resource_type = None
    #: Environment name to be used for logging.
    name = None
    #: OS environment variables.
    os_env = OSEnv()
    #: Environment platform
    platform = None
    #: use resource cache for this environment
    use_cache = False
    #: Collection of exclusive build cache directory lock holders
    exclusive_build_cache_locks = []

    def __init__(self, version=None, platform=None):
        cls = type(self)
        self.logger = logging.getLogger(cls.__module__.rsplit(".")[-1]).getChild(cls.__name__)
        self.sys_path_utils = []
        self.resource_id = None
        self.version = str(version) if version is not None else None
        if platform:
            self.platform = platform
        self.__environment_folder = None

    @common_patterns.classproperty
    def build_cache_dir(cls):
        """ Returns path to the directory where task can store build cache, which can be re-used by other task(s). """
        dst = common_config.Registry().client.tasks.build_cache_dir
        if not os.path.exists(dst):
            os.mkdir(dst, 0o755)
        return dst

    @classmethod
    def exclusive_build_cache_dir(cls, prefix, sequence=None):
        """
        Returns lowest sequence path to the directory where task can exclusively store build cache,
        which can be re-used by other task(s).
        I.e. for N parallel executing tasks following directories will be returned:
        - `{build_cache_dir}/{prefix}/{sequence 0}`
        - `{build_cache_dir}/{prefix}/{sequence 1}`
        - ...
        - `{build_cache_dir}/{prefix}/{sequence N-1}`

        :param prefix:      Sub-directory name where exclusive directory will be searched or created.
        :param sequence:    Sequence generator. `xrange` will be used by default.
        """
        def _dir_maker(d):
            try:
                os.mkdir(d, 0o755)
            except OSError as ex:
                if ex.errno != errno.EEXIST:
                    raise
            return d

        cdir = common_config.Registry().client.tasks.build_cache_dir
        dst = os.path.join(cdir, prefix)
        for d in (cdir, dst):
            _dir_maker(d)

        logging.info("Searching for exclusive build cache dir at %r", dst)
        for n in six.moves.map(str, sequence or (str(_).zfill(3) for _ in range(1000))):
            res = os.path.join(dst, n)
            logging.debug("Checking directory for sequence %r", n)
            try:
                cls.exclusive_build_cache_locks.append(common_threading.RFLock(res + ".lock").acquire(False))
                return _dir_maker(res)
            except IOError as ex:
                logging.info("Sequence %r locked already: %s", n, ex)
        raise IndexError("No free exclusively locked directory found with the sequence generator provided")

    @property
    def meta(self):
        return FileSizeMetadata(self.get_environment_folder())

    @staticmethod
    def update_os_env(key, value, prepend=True):
        os.environ[key] = os.pathsep.join(
            filter(None, (tuple if prepend else reversed)((value, os.environ.get(key))))
        )

    @staticmethod
    def set_os_env(key, value):
        os.environ[key] = str(value) if value is not None else ""

    @staticmethod
    def update_os_path_env(value, prepend=True, key="PATH"):
        """ Mostly used to start 3rd-party binaries subprocessess. Additionally cut-off skynet binaries path. """
        os.environ[key] = common_os.path_env(value, prepend, key)

    @abc.abstractmethod
    def prepare(self):
        """
        Prepares the environment - downloads and extracts required files, modifies `os.environ` and so on.

        :return: :class:`string` with full path to the environment's directory.
        """
        pass

    def touch(self):
        """
        Touch the environment.
        """
        self.meta.touch()

    @property
    def released_compatible_resource(self):
        return self.get_released_compatible_resource()

    def get_released_compatible_resource(self, allow_binary_compatible=False):
        """
        Determines latest releases resource for given object, based on `resource_type`, `version` object attributes
        and current platform. In case of no compatible resource found, `None` will be returned.

        :return: :class:`dict` with the resource information or `None`
        """
        current_platform = legacy.current_task and legacy.current_task.platform or common_platform.platform()
        self.logger.debug(
            "Try to find a released resource with type '%s', platform '%s', version '%s'",
            self.resource_type, current_platform, self.version
        )
        from sandbox import sdk2
        attrs_query = {"released": ctt.ReleaseStatus.STABLE}
        if self.version:
            attrs_query["version"] = str(self.version)
        resources = sdk2.Resource.find(
            type=self.resource_type,
            state=ctr.State.READY,
            attrs=attrs_query
        ).limit(100)
        for resource in resources:
            if self.check_resource_platform(resource, allow_binary_compatible):
                return resource
        return None

    def _environment_resource_query(self):
        """
        Builds a filter to be used by :meth:`find_environment_resource`

        :return: :class:`dict` to be passed to GET /api/v1.0/resource method.
        """
        query = {
            "type": str(self.resource_type),
            "state": ctr.State.READY,
        }
        if self.version:
            query["attrs"] = {"version": self.version}

        return query

    def _environment_resource_query_wrapper(self):
        # TODO: SANDBOX-6513
        query = self._environment_resource_query()
        if "resource_type" not in query:
            return query
        sdk2_query = {
            "type": query["resource_type"],
            "state": query["status"],
            "attrs": query["all_attrs"]
        }

        return sdk2_query

    @property
    def cached_resource_id(self):
        """
        Returns compatible resource id using a quick service API call

        :return: resource id, or None
        """
        current_platform = legacy.current_task and legacy.current_task.platform or common_platform.platform()
        params = {"type": self.resource_type, "platform": current_platform}
        if self.version:
            params["version"] = self.version
        from sandbox import sdk2
        result = sdk2.Task.current.server.service.resources.read(**params)
        if result:
            self.logger.debug("Found cached resource id #%s for params %s", result, params)
            self.resource_id = result
            return result
        self.logger.debug("Did not find cached resource for params %s", params)

    @property
    def compatible_resource(self):
        """
        Find appropriate resource compatible with the current platform.

        :raises SandboxEnvironmentError: if such resource not find
        :return: path to the synced resource
        """
        query = self._environment_resource_query_wrapper()
        from sandbox import sdk2
        environment_resources = list(sdk2.Resource.find(**query).limit(100))
        msg = "Cannot find any '{}' '{}' resource for Sandbox environment (query: {!r}).".format(
            self.resource_type, self.version, query
        )

        if not environment_resources:
            raise common_errors.SandboxEnvironmentError(msg)
        self.logger.debug("Environment resources: %r", environment_resources)
        for resource in environment_resources:
            if self.check_resource_platform(resource):
                self.resource_id = resource.id
                return resource
        if self.version is not None:
            msg += " There is no such version for current platform"
        raise common_errors.SandboxEnvironmentError(msg)

    def get_environment_resource(self):
        from sandbox import sdk2
        if sdk2.Task.current is None:
            raise common_errors.TaskError("It's forbidden to install environments in server-side methods")
        resource_id = (self.use_cache and self.cached_resource_id) or self.compatible_resource.id
        return str(sdk2.ResourceData(sdk2.Resource[resource_id]).path)

    def check_resource_platform(self, resource, allow_binary_compatible=False):
        """
        Checks, whether given resource is compatible with the current platform.

        :param resource: Resource object to be checked.
        :param allow_binary_compatible: Allow other, binary compatible platforms if `True`
        :return: `True`, if the given resource is compatible with the current platform, `False` otherwise.
        :rtype: bool
        """
        resource_platform = getattr(resource, "platform", "")
        if resource_platform == "any":
            return True
        current_platform = (
            self.platform or
            legacy.current_task and legacy.current_task and legacy.current_task.platform or
            common_platform.platform()
        )
        resource_platforms = resource_platform.split(',')
        self.logger.debug("Check platforms '%r'", resource_platforms)
        for resource_platform in resource_platforms:
            self.logger.debug(
                "Check resource %s, platform '%s' with local platform '%s'",
                resource.id, resource_platform, current_platform
            )
            if not allow_binary_compatible:
                if common_platform.compare_platforms(resource_platform, current_platform):
                    return True
            else:
                if common_platform.is_binary_compatible(resource_platform, current_platform):
                    return True
        return False

    def get_environment_folder(self):
        """
        Returns full path to the environment's location directory.

        :return: :class:`string` with full path to the environment's location directory.
        """
        if self.__environment_folder:
            return self.__environment_folder

        if self.resource_id:
            folder_name = "{0}_{1}_res{2}".format(self.name, self.version, self.resource_id)
        else:
            folder_name = "{0}_{1}".format(self.name, self.version)
        candidate = os.path.abspath(os.path.join(common_config.Registry().client.tasks.env_dir, folder_name))

        # there may be no access: e.g. for privileged task we mount filesystem as read-only
        if os.access(candidate, os.W_OK):
            self.__environment_folder = candidate
        else:
            new_candidate = os.path.abspath(str(task.Task.current.path(folder_name)))
            self.logger.info(
                "There is no write permission to the directory '%s', creating environment at '%s'",
                candidate,
                new_candidate
            )
            self.__environment_folder = new_candidate
        return self.__environment_folder

    def extract_tar(self, tar_file, destination):
        """
        Extracts given tarball file into the given directory.

        :param tar_file:    tarball file name with full path.
        :param destination: directory name with full path to extract the given tarball to.
        :return:            given directory name.
        """
        self.logger.debug("Extract tarball file '%s' to '%s'", tar_file, destination)
        tarfile.open(tar_file, "r:*").extractall(destination)
        return destination

    def check_environment(self):
        """
        Checks if the environment was prepared and works fine.
        Default implementation only checks whether :py:attr:`sys_path_utils` is in system path.

        :return: `True` in case the environment is OK
        :raises common.errors.SandboxEnvironmentError: the check has failed
        """

        for tool in self.sys_path_utils:
            tool_path = paths.which(tool)
            if tool_path is None:
                raise common_errors.SandboxEnvironmentError("Cannot find '{}' util in the system path.".format(tool))
            else:
                self.logger.debug('Sandbox environment util "%s" path: %s', tool, tool_path)
        return True

    def check_tool_pathes(self, tool_pathes):
        """
        Checks if the tools were prepared and work fine.
        Default implementation just checks whether specified paths exist.

        :return: `True` in case specified tools are OK
        :raises common.errors.SandboxEnvironmentError: the check has failed.
        """
        for tool_path in tool_pathes:
            if os.path.exists(tool_path):
                self.logger.debug("Sandbox environment util '%s' exists.", tool_path)
            else:
                raise common_errors.SandboxEnvironmentError(
                    "Sandbox environment util '{}' does not exist.".format(tool_path)
                )
        return True


class PipEnvironment(SandboxEnvironment):
    """
    Base class to install python packages through pip (from skynet).

    Can install python packages to current environment or to specific virtual environment.
    To use virtual environment, pass a into constuctor an object with these properties
    (:py:class:`sandbox.sandbosdk.environments.VirtualEnvironment` object is the best fit):

        :executable: path to executable python interpreter
        :user_site: property which reflects the availability of user site-packages from the virtual environment
    """

    PIP_VERSION = "9.0.1"
    PYPI_URLS = [
        "https://pypi.yandex-team.ru/simple/",
    ]
    # timeout for pip to wait for reply from package index, seconds
    INDEX_TIMEOUT = 60

    def __init__(
        self,
        package_name,
        version=None,
        import_name=None,
        use_pre=False,
        index_url=None,
        use_wheel=False,
        custom_parameters=None,
        venv=None,
        env_update=True,
        use_user=None
    ):
        """
        :param package_name: Name of necessary python-package.
        :param version: Version of package. Optional argument. If not specified will install most
                        recent version.
        :param import_name: Package name to import during check. Optional argument.
                            If not specified will use package name
        :param use_pre: Use or not pre version of package
        :param index_url: Pass or not Base URL of Python Package Index. Will ignore this parameter if
                          wheel mode used
        :param use_wheel: Use or not wheel mode. In wheel mode install package from prepared wheel archive.
        :param venv: Install package into selected virtual environment (VirtualEnvironment object)
        :param env_update: Allow PipEnvironment to update the environment's common packages when it is
                           required. For example, 'wheel' package installation will fail with older versions
                           of 'pip' (< 1.5) and 'setuptools' (< 0.8).
        :param use_user: Allow PipEnvironment to install packages to user site-packages directory
                         (~/.local/ by default).
                         By default value of this parameter is calculated depending on venv parameter:
                            default value of use_user is True when no venv is given;
                            default value of use_user is False when venv is given.
        """
        super(PipEnvironment, self).__init__(version)
        self.name = package_name
        self.use_pre = use_pre
        if index_url and index_url not in self.PYPI_URLS:
            self.PYPI_URLS = [index_url] + self.PYPI_URLS
        self.import_name = import_name or package_name
        self.use_wheel = use_wheel
        self.custom_parameters = custom_parameters if custom_parameters is not None else []
        self.venv = venv
        self.env_update = env_update

        if use_user is None:
            self.use_user = venv is None
        else:
            self.use_user = use_user

        self.build_dir = os.path.expanduser("~/.local/build")

    def _environment_resource_query(self):
        query = super(PipEnvironment, self)._environment_resource_query()
        query.setdefault("attrs", {}).update({"name": self.name})
        return query

    def _get_module_version(self, module_name):
        """
        Get version of given module

        :param module_name: name of module to get version

        :rtype: bool or None
        :return: module version (pkg_resources.parse_version call result) or 'None' when module is not installed
                 yet.
        """

        if self.venv:
            module_version = self.venv.get_module_version(module_name)
        else:
            module = self._import_module(module_name)
            if module:
                module_version = pkg_resources.parse_version(module.__version__)
            else:
                module_version = None

        logging.debug("Target environment's package '{0}' version is '{1}'".format(module_name, module_version))

        return module_version

    @common_context.skip_if_binary("PipEnvironment._install_package")
    def _install_package(self, name, version=None):
        """
        Installs package into selected environment (virtual, when self.venv is defined, or current otherwise)

        :param name: package name
        :param version: package version

        :rtype: None
        """
        self.clear_build_cache()

        if version:
            pip_package_name = name + "==" + version
        else:
            pip_package_name = name

        self.logger.debug("Installing %s python package", pip_package_name)

        if self.venv is None:
            py_executable = sys.executable
        else:
            py_executable = self.venv.executable

        pip_executable = os.path.join(os.path.dirname(py_executable), "pip")

        parameters = [
            py_executable, pip_executable,
            "install", "--upgrade",
            "--build={}".format(self.build_dir),
            "-vvv",
        ]

        if self._get_module_version("pip") > pkg_resources.parse_version("8.0"):
            parameters.append("--disable-pip-version-check")

        # User site-packages can be not available in some virtual environments.
        # So, when virtual environment had been created with no "--system-site-packages"
        # option, pip install --user will fail.
        # Check, that user site-packages are available from target virtual environment or
        # from current environment, when no target given.
        if self.venv:
            user_site_enabled = self.venv.user_site
        else:
            site_module = importlib.import_module("site")
            user_site_enabled = site_module.ENABLE_USER_SITE

        # Allow pip to use user site-packages when it is allowed.
        if user_site_enabled and self.use_user:
            parameters.append("--user")

        if self.use_wheel:
            self.resource_type = "PYTHON_WHEEL"
            parameters.append("--no-index")
            parameters.append("--find-links={}".format(self.get_environment_resource()))

        parameters.append(" ".join(map("'{}'".format, pip_package_name.split())))
        parameters += self.custom_parameters
        if self.use_pre:
            parameters.append("--pre")

        env = os.environ.copy()
        env.update(
            VirtualEnvironment.get_extra_build_variables(
                fix_rpath=False,
                use_cflags_ldflags=True
            )
        )
        installed = False
        for pypi_url in self.PYPI_URLS:
            cmd = parameters + [
                "--index-url {url}".format(url=pypi_url),
                "--timeout", str(self.INDEX_TIMEOUT),
            ]
            for attempt in [1, 2, 3]:
                self.logger.info("Trying `%s` index url, attempt %s", pypi_url, attempt)
                pl = helpers.ProcessLog(
                    logger="pip_install_{0}".format(re.sub(r"[^\w\-]+", "_", pip_package_name))
                )
                with pl:
                    pip_proc = helpers.process.subprocess.Popen(
                        " ".join(map(str, cmd)), env=env, stdout=pl.stdout, stderr=pl.stderr, shell=True
                    )
                    pip_proc.wait()

                if pip_proc.returncode == 0:
                    installed = True
                    break

                self.logger.error("Error occurred, see '%s' for details", pl.stdout.path)
                err_contents = open(str(pl.stdout.path)).read()
                self.logger.warning("Removing pip's build cache directory for package %s", name)
                self.clear_build_cache(name)
                if "INTERNAL SERVER ERROR" in err_contents or "502 Server Error: Bad Gateway" in err_contents:
                    # see e.g. SEARCH-2570
                    self.logger.info("PyPi server made a boo-boo, wait some time and retry")
                    time.sleep(30 + 30 * (attempt - 1) * random.random())
                    continue

            if installed:
                break
        else:
            raise PipError("Can not install {}. See logs for more info.".format(pip_package_name))
        local_bin_path = os.path.expanduser("~/.local/bin")
        self.update_os_env("PATH", local_bin_path)

    def _import_module(self, module_name):
        """
        Try to import module. Return module object on success.

        :param module_name: name of module to import

        :rtype: module
        :return: module object (for successful import) or None (otherwise)
        """
        try:
            module = __import__(module_name, globals(), locals())
        except ImportError as error:
            self.logger.error(
                "Cannot import %s for %s environment. Error: %s",
                module_name, self.name, error
            )
            return None

        return module

    def touch(self):
        pass

    @common_context.skip_if_binary("PipEnvironment.clear_build_cache")
    def clear_build_cache(self, package=None):
        """
        Removes directory with package build cache (sources, object files, etc.)
        When no package name given, removes full build cache directory.

        Sometimes new package build fails in case of existance of package source code files,
        remained from previous unsuccessfull build. This method allows you to cleanup the build cache
        directory from such files of a specific package.

        :param package: package build cache directory name to remove.

        :rtype: NoneType
        :return: None
        """
        if package:
            remove_path = os.path.join(self.build_dir, package)
        else:
            remove_path = self.build_dir

        try:
            shutil.rmtree(remove_path)
        except OSError as e:
            # Ignore 'No such file or directory' errors
            if e.errno != errno.ENOENT:
                raise e

    @common_context.skip_if_binary("PipEnvironment.update_environment")
    def update_environment(self):
        """
        Update environment. Install/update some packages when it is necessary.
        This function does as little as possible. It will not update packages when it is not required
        by conditions, given in constructor.
        """
        if self.use_wheel:
            try:
                current_use_user = self.use_user

                self.use_wheel = False
                self.use_user = False

                if self._get_module_version("pip") < pkg_resources.parse_version("1.5"):
                    logging.warning("pip version is too old for wheel installation. Updating pip.")
                    self._install_package("pip", version=self.PIP_VERSION)
                    logging.debug("New pip version: %s", self._get_module_version("pip"))

                if self._get_module_version("setuptools") < pkg_resources.parse_version("0.8"):
                    logging.warning("setuptools version is too old for wheel installation. Updating setuptools.")
                    self._install_package("setuptools")
                    logging.debug("New setuptools version: %s", self._get_module_version("setuptools"))
            finally:
                self.use_wheel = True
                self.use_user = current_use_user

    @common_context.skip_if_binary("PipEnvironment.prepare")
    def prepare(self):
        """
        Install the package to the environment.
        """
        if self.env_update:
            self.update_environment()

        self._install_package(self.name, self.version)

    @common_context.skip_if_binary("PipEnvironment.check_environment")
    def check_environment(self):
        """
        Try to do import

        :return: on success return True, else False
        :rtype: bool
        """
        if self.venv:
            cmd = [self.venv.executable, "-c", "import {}".format(self.import_name)]
            with helpers.ProcessLog(logger="test_venv_import_{}".format(self.import_name)) as pl:
                p = sp.Popen(cmd, stdout=pl.stdout, stderr=pl.stderr)
                p.wait()
            return p.returncode == 0
        else:
            return bool(self._import_module(self.import_name))


class TarballToolkitBase(SandboxEnvironment):

    def validate(self):
        if not self.meta.exists:
            return False

        absent, extra, different = self.meta.check_files()

        if absent or different:
            logging.info(
                "Environment '%s' is broken. Absent: %s. Different size (stored, real): %s",
                self.name, list(absent), list(different)
            )
            return False
        return True

    def prepare(self):
        resource_path = self.get_environment_resource()
        env_dir = self.get_environment_folder()
        lock_file = os.path.join(os.path.dirname(env_dir), os.path.basename(env_dir) + ".lock")
        lock = (
            common_threading.RFLock(lock_file)
            if ctc.Tag.MULTISLOT in common_config.Registry().client.tags else
            common_context.NullContextmanager()
        )

        with lock:
            if os.path.exists(env_dir):
                if self.validate():
                    self.logger.info("Environment folder already exists at '%s'", env_dir)
                    return env_dir
                else:
                    # Env is broken try create it again
                    logging.info("Deleting environment '%s'.", self.name)
                    shutil.rmtree(env_dir)

            self.extract_tar(resource_path, env_dir)
            self.meta.store()

        return env_dir


class SvnEnvironment(SandboxEnvironment):
    """ Svn binary built from `arcadia/contrib/libs/subversion/subversion/svn` """
    resource_type = "SVN_BINARY"
    sys_path_utils = ["svn"]
    use_cache = True

    STABLE = "1.9.5"

    def __init__(self, version=None, platform=None):
        super(SvnEnvironment, self).__init__(version or self.STABLE, platform)

    def prepare(self):
        svn_binary_path = self.get_environment_resource()
        self.update_os_path_env(os.path.dirname(svn_binary_path))
        self.check_environment()
        return svn_binary_path

    def touch(self):
        # Don't create environment directory, synced resource is enough
        pass


class ArcEnvironment(TarballToolkitBase):
    """ Arc client """
    resource_type = "ARC_CLIENT"
    name = "arc"
    arc_secret = "arc_secret"
    yav_default_key = "arc_token"
    yav_token_name = "yav_token"
    sys_path_utils = ["arc", ]
    use_cache = True
    __common_arc_token = None

    YAV_TOKEN_ENV = "YAV_TOKEN"

    def __init__(self, token_name="ARC_TOKEN", token_owner=None, token=None, yav_token=None, export_token_env=True):
        """
        The constructor takes either name and owner or a token itself
        :param token_name: string, contains arc token name
        :param token_owner: string, contains a login of the token owner; if the owner is not listed
            it's being taken the task's owner during the token obtaining
        :param token: string, the token itself, if passed, it's always used instead of obtaining it
            using the name and the owner.
        :param yav_token: OAuth token for YAV
        """
        super(ArcEnvironment, self).__init__()
        self.token_name = token_name
        self.token_owner = token_owner
        self.__token = token
        self.__yav_token = yav_token
        self._arc_binary_path = None
        self._export_token_env = export_token_env

    @common_patterns.singleton_classproperty
    def common_arc_token(cls):
        return cls.__common_arc_token

    @classmethod
    def set_common_arc_token(cls, value):
        cls.__common_arc_token = value

    @property
    def compatible_resource(self):
        # ARC_CLIENT resource is available for 3 platforms named `linux`,
        # `darwin` and `win32`.
        if sys.platform.startswith("linux"):
            current_platform = "linux"
        else:
            current_platform = sys.platform
        self.logger.debug(
            "Try to find a released resource with type '%s', platform '%s'",
            self.resource_type, current_platform
        )
        from sandbox import sdk2
        res = sdk2.Resource.find(
            type=self.resource_type,
            state=ctr.State.READY,
            hidden=0,
            limit=1,
            omit_failed=True,
            attrs={"platform": current_platform, "released": ctt.ReleaseStatus.STABLE},
            owner="ARC",
        ).first()

        if res is None:
            self.logger.warning("Found no Arc resource")
            return None

        self.logger.debug("found res %s", res)
        self.resource_id = res.id
        return res

    def _get_token(self, token_name, default_key):
        from sandbox import sdk2
        try:
            ctx_token = getattr(sdk2.Task.current.Context, token_name)
            param = (
                ctx_token and sdk2.yav.Secret.__decode__(ctx_token)
                if isinstance(sdk2.Task.current, sdk2.task.OldTaskWrapper) else
                getattr(sdk2.Task.current.Parameters, token_name, None)
            )

            if param:
                self.logger.info("Try to get token with secret %s", param)
                return param.data()[
                    param.default_key or default_key
                ]
        except Exception as e:
            self.logger.error(
                "Caught exception while finding yav arc token", exc_info=e
            )

    @property
    def _arc_token(self):
        """
        Uses token_owner from init or gets the owner of the current task, find token for arc with `token_name` in
        owner's vault and returns it.
        """
        if self.__token is not None:
            return self.__token

        self.__token = self._get_token(self.arc_secret, self.yav_default_key)
        if self.__token is not None:
            return self.__token

        from sandbox import sdk2
        owner = self.token_owner or sdk2.task.Task.current.owner
        try:
            self.logger.debug("Getting %s for %s from vault", self.token_name, owner)
            self.__token = sdk2.Vault.data(owner, self.token_name)
            return self.__token
        except common_errors.VaultError as e:
            self.logger.error(
                "Caught exception %r while finding token `%s` for group %s. Try to use common token",
                e, self.token_name, owner,
            )
            self.__token = os.environ.get("ARC_TOKEN")
            if not self.__token:
                self.__token = self.common_arc_token
            return self.__token

    @property
    def _yav_token(self):
        if self.__yav_token is None:
            self.__yav_token = self._get_token(self.yav_token_name, self.yav_token_name)
        return self.__yav_token

    def prepare(self):
        env_dir = super(ArcEnvironment, self).prepare()
        self._arc_binary_path = os.path.join(env_dir, "arc")
        self.update_os_path_env(os.path.dirname(self._arc_binary_path))
        if self._export_token_env:
            for k, v in self.token_env.items():
                self.logger.debug("Adding %s to environment variables", k)
                self.set_os_env(k, v)
        else:
            self.logger.debug("Will not add arc tokens to environment")
        self.check_environment()
        return self._arc_binary_path

    @property
    def arc_binary_path(self):
        if self._arc_binary_path is None:
            return self.prepare()

        return self._arc_binary_path

    @property
    def token_env(self):
        res = dict()
        if self._arc_token is not None:
            res["ARC_TOKEN"] = self._arc_token
        if self._yav_token is not None:
            res["YAV_TOKEN"] = self._yav_token
        return res


class SandboxHgEnvironment(TarballToolkitBase):
    """ Statically built Mercurial client """
    resource_type = "HG_TOOLKIT"
    name = "hg"
    sys_path_utils = ["hg", ]
    use_cache = True

    STABLE = "3.0.9y"

    def __init__(self, version=None, **kws):
        super(SandboxHgEnvironment, self).__init__(version or self.STABLE, **kws)

    def prepare(self):
        env_dir = super(SandboxHgEnvironment, self).prepare()
        self.update_os_path_env(os.path.join(env_dir, "hg"))
        self.check_environment()
        return env_dir


class GCCEnvironment(TarballToolkitBase):
    """ GCC Toolkit environment. """
    resource_type = 'GCC_TOOLKIT'
    name = "gcc"
    c_compiler_name = "gcc"
    cxx_compiler_name = "g++"
    use_cache = True

    def __init__(self, version="4.8.2"):
        major = version.rsplit(".", 1)[0]
        self.c_compiler_name += "-{}".format(major)
        self.cxx_compiler_name += "-{}".format(major)
        super(GCCEnvironment, self).__init__(version)

    @property
    def compatible_resource(self):
        current_platform = (
            legacy.current_task and legacy.current_task and legacy.current_task.platform or common_platform.platform()
        )
        self.logger.info(
            "Searching for released '%s' environment, version '%s', platform '%s'",
            self.name, self.version, current_platform
        )
        resource = self.get_released_compatible_resource(allow_binary_compatible=True)
        if not resource:
            self.logger.info(
                "Found no released '%s' resource with version '%s' for platform '%s'. "
                "Search for not released resource for this version.",
                self.resource_type, self.version, current_platform
            )
            resource = super(GCCEnvironment, self).compatible_resource

        self.resource_id = resource.id
        return resource

    def _set_compiler_environment_variables(self, env_dir):
        self.update_os_env("LD_LIBRARY_PATH", os.path.join(env_dir, self.name, "lib"))
        self.update_os_env("LD_RUN_PATH", os.path.join(env_dir, self.name, "lib"))
        # COMPILER_PATH is needed by clang to choose right "ld"
        self.update_os_env("COMPILER_PATH", os.path.join(env_dir, self.name, "bin"))

    def get_compilers(self):
        env_dir = self.get_environment_folder()

        c_compiler_path = os.path.join(env_dir, self.name, "bin", self.c_compiler_name)
        cxx_compiler_path = os.path.join(env_dir, self.name, "bin", self.cxx_compiler_name)
        self._set_compiler_environment_variables(env_dir)
        compilers = {"CC": c_compiler_path, "CXX": cxx_compiler_path}
        self.check_tool_pathes(list(compilers.values()))
        return compilers

    def prepare(self):
        env_dir = super(GCCEnvironment, self).prepare()
        os.environ.update(self.get_compilers())
        return env_dir


class GDBEnvironment(TarballToolkitBase):

    name = "gdb"
    resource_type = "GDB_TOOLKIT"
    use_cache = True

    def get_released_compatible_resource(self, allow_binary_compatible=False):
        current_platform = legacy.current_task and legacy.current_task.platform or common_platform.platform()
        self.logger.debug(
            "Try to find a released resource with type '%s', platform '%s', version '%s'",
            self.resource_type, current_platform, self.version
        )
        from sandbox import sdk2
        for resource in sdk2.Resource.find(
            type=self.resource_type,
            state=ctr.State.READY,
            attrs={"released": ctt.ReleaseStatus.STABLE},
        ).limit(100):
            if self.version:
                if str(resource.version) != self.version:
                    continue
            if self.check_resource_platform(resource, allow_binary_compatible):
                return resource
        return None

    @property
    def compatible_resource(self):
        current_platform = legacy.current_task and legacy.current_task.platform or common_platform.platform()
        resource = self.released_compatible_resource
        if not resource:
            raise common_errors.SandboxEnvironmentError(
                "Cannot find a released resource with type '{}', platform '{}' and version '{}'".format(
                    self.resource_type, current_platform, self.version
                )
            )
        self.resource_id = resource.id
        return resource

    def prepare(self):
        env_dir = super(GDBEnvironment, self).prepare()
        self.update_os_path_env(os.path.join(env_dir, "gdb", "bin"))
        return env_dir


class CMakeEnvironment(TarballToolkitBase):

    name = "cmake"
    resource_type = "CMAKE_EXECUTABLE"

    sys_path_utils = ["cmake"]

    def prepare(self):
        env_dir = super(CMakeEnvironment, self).prepare()
        self.update_os_path_env(os.path.join(env_dir, "bin"))
        self.check_environment()
        return env_dir


class IntelCompilerEnvironment(TarballToolkitBase):

    name = "icc"
    resource_type = "ICC_TOOLKIT"

    def __init__(self, *args, **kwargs):
        super(IntelCompilerEnvironment, self).__init__(*args, **kwargs)
        self.directory = None

    def prepare(self):
        self.directory = super(IntelCompilerEnvironment, self).prepare()
        self.update_os_path_env(os.path.join(self.directory, "bin"))
        return self.directory


class CCache(SandboxEnvironment):
    name = "ccache"
    resource_type = "CCACHE_EXECUTABLE"

    CACHE_SIZE = "10G"

    def prepare(self):
        from sandbox import sdk2
        executable = str(sdk2.ResourceData(sdk2.Resource[self.compatible_resource]).path)

        settings = common_config.Registry()
        os.environ["CCACHE_DIR"] = os.path.join(self.build_cache_dir, ".ccache")
        os.environ["CCACHE_UNIFY"] = "1"
        os.environ["CCACHE_BASEDIR"] = os.path.dirname(settings.client.tasks.data_dir)
        os.environ["CCACHE_LOGFILE"] = sdk2.Task.current.log_path("ccache.log")

        with helpers.ProcessLog(logger="ccache") as pl:
            helpers.process.subprocess.check_call(
                [executable, "-M", self.CACHE_SIZE], stdout=pl.stdout, stderr=pl.stderr
            )

        return executable


class NodeJS(TarballToolkitBase):
    """ NodeJS with `npm` and `bower` tools embedded into it. """

    name = "NodeJS"
    resource_type = "NODEJS_PACKAGE"
    use_cache = True

    @property
    def bindir(self):
        """ Returns path to binary directory of the prepared environment. """
        return os.path.join(self.get_environment_folder(), "bin")

    @property
    def node(self):
        """ Returns path to `node` executable of the prepared environment. """
        return os.path.join(self.bindir, "node")

    @property
    def npm(self):
        """ Returns path to `npm` executable of the prepared environment. """
        return os.path.join(self.bindir, "npm")

    @property
    def compatible_resource(self):
        current_platform = legacy.current_task and legacy.current_task.platform or common_platform.platform()
        resource = self.released_compatible_resource
        if not resource and self.version:
            resource = super(NodeJS, self).compatible_resource
        if not resource:
            raise common_errors.SandboxEnvironmentError(
                "Cannot find a released resource with type '{}', platform '{}' and version '{}'".format(
                    self.resource_type, current_platform, self.version
                )
            )
        self.resource_id = resource.id
        return resource

    def prepare(self):
        basedir = super(NodeJS, self).prepare()
        self.update_os_path_env(os.path.join(basedir, "bin"))
        return basedir


class Golang(TarballToolkitBase):
    """ Golang with `go` tool embedded into it. """

    name = "Golang"
    resource_type = "GOLANG_PACKAGE"
    use_cache = True

    @property
    def bindir(self):
        """ Returns path to binary directory of the prepared environment. """
        return os.path.join(self.get_environment_folder(), "go", "bin")

    @property
    def gotool(self):
        """ Returns path to `go` executable of the prepared environment. """
        return os.path.join(self.bindir, "go")

    def prepare(self):
        basedir = super(Golang, self).prepare()
        self.update_os_path_env(os.path.join(basedir, "go", "bin"))
        return basedir


class Xcode(SandboxEnvironment):
    """ Xcode installation with developers dir. """

    STABLE = "11.2.1"

    def __init__(self, version):
        super(Xcode, self).__init__(version or self.STABLE, "darwin")

    def prepare(self):
        Xcode.prepare_xcode(self.version)

    @classmethod
    def prepare_xcode(cls, version=None):
        from sandbox import sdk2
        sdk2.Task.current.agentr.prepare_xcode(version or cls.STABLE)


class XcodeSimulator(SandboxEnvironment):
    """ Xcode Simulator installation. """

    def __init__(self, version):
        if version is None:
            raise common_errors.SandboxEnvironmentError("XcodeSimulator version is not specified")
        super(XcodeSimulator, self).__init__(version, "darwin")

    def prepare(self):
        XcodeSimulator.prepare_xcode_simulator(self.version)

    @classmethod
    def prepare_xcode_simulator(cls, version):
        from sandbox import sdk2
        sdk2.Task.current.agentr.prepare_xcode_simulator(version)
