from __future__ import absolute_import

# helper functions for GDB coredumps

import os
import time
import logging
import textwrap

from sandbox.common import rest as common_rest


def gdb_trace_command(binary_path, coredump_path):
    """
    Return command to get GDB trace from coredump.
    This command works only with gdb72.
    """
    print_backtraces_python_script = textwrap.dedent(
        """
        import gdb


        for n in [t.num for t in gdb.selected_inferior().threads()]:
            try:
                gdb.execute("thr {}".format(n))
                gdb.execute("where")
            except Exception as e:
                print(e)
            gdb.write("\\n")
        """
    )
    cmd = [
        "gdb",
        "--core", coredump_path,
        "--eval-command", "python {}".format(print_backtraces_python_script),
        "--batch", "--args", binary_path
    ]
    return cmd


def get_html_view_for_logs_file(name, file_name, resource, is_dir=True):
    """
    Get representation of resource as a link to be put in task.info

    :param name:        description of file within the resource (stderr, stdout, gdb_trace, etc.)
    :param file_name:   filename within the resource
    :param resource:    log(s) resource ID or object
    :param is_dir:      flags the given resource is a directory
    :return:            HTML with links to resource files
    :rtype: str
    """
    rest = common_rest.DispatchedClient()
    logging.debug("name, file_name, resource: %s, %s, %r", name, file_name, resource)
    file_resource_url = getattr(resource, "http_proxy", None)
    if not file_resource_url:
        resources = rest.resource.read(id=resource, limit=1)
        if resources["items"]:
            resource = resources["items"][0]
        else:
            return "Invalid resource {!r}".format(resource)

        file_resource_url = resource["http"]["proxy"]
    if is_dir:
        file_resource_url = "/".join((file_resource_url, str(file_name)))
    return """<b><font color="red">{}</font></b>: <a href="{}" target="_blank">{}</a>""".format(
        name, file_resource_url, file_name
    )


def get_core_path(coredump_dir, binary_name, process_pid):
    """
    Get path to coredump of a process with certain PID if it exists.

    The name of coredump should be {process_name}.{process_pid}[.signal]
    Note that {process_name} can be incomplete. Examples:
    # basesearch.27892.11
    # ranking_middlese.14692 (without signal value)

    :param coredump_dir: coredumps directory
    :param binary_name: name of binary file
    :param process_pid: PID of process
    :return: path to coredump, if it exists; otherwise return None
    """
    if not os.path.exists(coredump_dir):
        return None
    coredump_path = None
    for core_name in os.listdir(coredump_dir):
        core_name_info = core_name.split(".")
        core_process_name = core_name_info[0]
        process_pid_match_template = ".{}." if len(core_name_info) == 3 else ".{}"
        process_pid_match = process_pid_match_template.format(process_pid)
        if (core_process_name in binary_name) and process_pid_match in core_name:
            coredump_path = os.path.join(coredump_dir, core_name)
    if coredump_path:
        # wait until coredump is written completely (size doesn't change for a while)
        coredump_is_done = False
        while not coredump_is_done:
            current_size = os.path.getsize(coredump_path)
            time.sleep(2)
            new_size = os.path.getsize(coredump_path)
            coredump_is_done = current_size == new_size
        return coredump_path
    else:
        return None


def filter_traceback(task_id, filter_dir, traceback_path, output_html):
    """
    Filter gdb traceback to html file

    :return created filtered traceback resource id
    :rtype int
    """
    from sandbox.sdk2.helpers import coredump_filter

    with open(output_html, "wb") as filtered_f:
        try:
            coredump_filter.get_stackdump(
                file_name=traceback_path,
                sandbox_failed_task_id=str(task_id),
                output_stream=filtered_f,
            )
        except Exception as exc:
            logging.error("Traceback filter failed with exception %s.", exc)
