from __future__ import absolute_import

import six
import functools as ft

from sandbox.common import format as common_format


class Type(type):
    @staticmethod
    def __is_valid_name(name):
        return isinstance(name, six.string_types) and name.isupper()

    def __new__(mcs, name, bases, namespace):
        forced_name = namespace.pop("name", None)
        assert not forced_name or mcs.__is_valid_name(forced_name), \
            "Invalid attribute's 'name' value in class '{}': '{}'".format(name, forced_name)
        return type.__new__(
            type(
                mcs.__name__, (mcs,), dict(name=property(lambda _, n=forced_name or common_format.ident(name): n))
            ),
            name, bases, namespace
        )

    @property
    def name(cls):
        return None


def memoized(func):
    ret = []

    @ft.wraps(func)
    def wrapper(*args, **kws):
        if ret:
            return ret[0]
        ret.append(func(*args, **kws))
        return ret[0]
    return wrapper


# noinspection PyPep8Naming
class dual_method(object):
    def __init__(self, method):
        self.__method = method

    def __get__(self, obj, type_):
        if obj is not None:
            return self.__method.__get__(obj, type_)
        type_of_type = type(type_)
        return getattr(type_of_type, self.__method.__name__).__get__(type_, type_of_type)


# noinspection PyPep8Naming
class overridable(object):
    """
    @DynamicAttrs
    """
    def __init__(self, method):
        self.method = method


class Query(object):
    """ REST API query """
    TRIAL_SIZE = 100
    MAX_LIMIT = 3000

    _limit = None
    _offset = 0
    _order = None

    class SortedField(property):
        def __init__(self, fget=None, fset=None, fdel=None, doc=None):
            self.__name__ = fget.__name__
            super(Query.SortedField, self).__init__(fget, fset, fdel, doc)

        def __repr__(self):
            return self.__name__

        def __neg__(self):
            ret = type(self)(self.fget, self.fset, self.fdel, self.__doc__)
            name = self.__name__
            ret.__name__ = name[1:] if name.startswith("-") else "-" + name
            return ret

        def __pos__(self):
            return self

    def __init__(self, base, restore, **constraints):
        self._base = base
        self._restore = restore
        self._constraints = constraints

    def __call__(self, limit=None, offset=None):
        assert limit is not None or self._limit is not None, "limit must be explicitly specified by method `limit()`"

        offset = self._offset if offset is None else offset

        if limit is None:
            limit = self._limit

        if not limit:
            trial = self._base[self._constraints, offset:self.TRIAL_SIZE:self._order]
            if trial["total"] <= self.TRIAL_SIZE:
                return trial
            limit = min(trial["total"], self.MAX_LIMIT)

        return self._base[
            self._constraints,
            offset:limit:self._order
        ]

    def __iter__(self):
        for item in self()["items"]:
            yield self._restore(item)

    @property
    def count(self):
        return self._base[self._constraints, 0:0:self._order]["total"]

    @property
    def __copy(self):
        query = type(self)(self._base, self._restore)
        query._constraints = self._constraints
        query._limit = self._limit
        query._offset = self._offset
        query._order = self._order
        return query

    def limit(self, value):
        query = self.__copy
        query._limit = int(value)
        return query

    def offset(self, value):
        query = self.__copy
        query._offset = int(value)
        return query

    def order(self, *values):
        for value in values:
            assert isinstance(value, self.SortedField), "cannot order by {!r}".format(value)
        query = self.__copy
        query._order = ",".join(map(str, values))
        return query

    def first(self):
        return next(iter(self.limit(1)), None)
